package com.xebialabs.plugin.manager.repository.nexus

import com.xebialabs.plugin.manager.config.{Credentials, Proxy, Server}
import com.xebialabs.plugin.manager.repository.config.PluginServerConfig
import org.apache.pekko.http.scaladsl.ClientTransport
import org.apache.pekko.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import org.apache.pekko.http.scaladsl.model.Uri
import org.apache.pekko.http.scaladsl.model.headers.{BasicHttpCredentials, HttpCredentials}
import spray.json._

import java.net.InetSocketAddress
import scala.util.{Failure, Success, Try}


case class NexusServerConfig(override val uri: Uri, override val credentials: HttpCredentials, override val transport: ClientTransport = ClientTransport.TCP)
  extends PluginServerConfig(uri, credentials, transport) {

  override def serviceUri: Uri = uri.withPath(uri.path / "service" / "local")

}

object NexusServerConfig {
  def apply(uri: Uri, credentials: HttpCredentials, proxyUri: Uri, proxyCredentialsOpt: Option[HttpCredentials]): NexusServerConfig = {
    val proxyAddress = new InetSocketAddress(proxyUri.authority.host.address(), proxyUri.authority.port)
    val transport = proxyCredentialsOpt match {
      case None => ClientTransport.httpsProxy(proxyAddress)
      case Some(proxyCredentials) => ClientTransport.httpsProxy(proxyAddress, proxyCredentials)
    }
    NexusServerConfig(uri, credentials, transport)
  }

  def fromConfig(serverConfig: Server): Try[NexusServerConfig] = {
    // validations first
    if (!"nexus".equals(serverConfig.getServerType))
      return Failure(new IllegalArgumentException(s"Plugin server ${serverConfig.getName} must have a 'nexus' server type"))

    if (serverConfig.getUrl == null || Try(Uri.apply(serverConfig.getUrl)).isFailure)
      return Failure(new IllegalArgumentException(s"Invalid url for plugin server ${serverConfig.getName}"))

    val credentials = serverConfig.getCredentials
    if (credentials == null || credentials.getPassword == null || credentials.getUsername == null)
      return Failure(new IllegalArgumentException(s"Both username and password must be present in the '${serverConfig.getName}' plugin server configuration"))

    if (serverConfig.getProxy != null && !isValid(serverConfig.getProxy))
      return Failure(new IllegalArgumentException(s"Invalid proxy configured for plugin server '${serverConfig.getName}''"))

    // creating a server config
    val httpCredentials = new BasicHttpCredentials(credentials.getUsername, credentials.getPassword)
    val uri = Uri.apply(serverConfig.getUrl)
    getProxyTransport(serverConfig) match {
      case Some(proxy) => Success(new NexusServerConfig(uri, httpCredentials, proxy))
      case None => Success(new NexusServerConfig(uri, httpCredentials))
    }
  }

  def getProxyTransport(serverConfig: Server): Option[ClientTransport] = {
    if (serverConfig.getProxy != null) {
      val proxy = serverConfig.getProxy
      val address = new InetSocketAddress(proxy.getHost, proxy.getPort)
      if (proxy.getCredentials != null && hasValidCredentials(proxy.getCredentials)) {
        val credentials = serverConfig.getProxy.getCredentials
        Some(ClientTransport.httpsProxy(address, new BasicHttpCredentials(credentials.getUsername, credentials.getPassword)))
      } else {
        Some(ClientTransport.httpsProxy(address))
      }
    } else {
      None
    }
  }

  def isValid(proxy: Proxy): Boolean = {
    val validCredentials = proxy.getCredentials == null || proxy.getCredentials.getUsername != null && proxy.getCredentials.getPassword != null
    val validAddress = proxy.getHost != null && proxy.getPort != null
    validCredentials && validAddress
  }

  def hasValidCredentials(credentials: Credentials): Boolean = credentials.getPassword != null && credentials.getUsername != null

  trait Protocol extends SprayJsonSupport with DefaultJsonProtocol {
    implicit val nexusServerConfigWriter: RootJsonWriter[NexusServerConfig] = {
      case NexusServerConfig(uri, _, _) =>
        JsObject(
          "server-type" -> "nexus".toJson,
          "url" -> uri.toString.toJson
        )
    }
  }
}

