package com.xebialabs.plugin.manager.repository

import com.typesafe.config.{Config, ConfigFactory}
import com.xebialabs.plugin.manager.PluginId
import com.xebialabs.plugin.manager.exception.PluginRepositoryException._
import com.xebialabs.plugin.manager.metadata.{ArtifactId, ExtendedMetadata, MetadataEntry}
import com.xebialabs.plugin.manager.repository.config.ListMethod.ByGroupId
import com.xebialabs.plugin.manager.repository.config.{ListMethod, PluginRepositoryConfig}
import com.xebialabs.plugin.manager.repository.protocol.ArtifactResult
import com.xebialabs.plugin.manager.repository.storage.PluginMetadataStorage
import grizzled.slf4j.Logging
import org.apache.pekko.actor.ActorSystem
import org.apache.pekko.http.scaladsl.model.{HttpRequest, HttpResponse, Uri}
import org.apache.pekko.stream.{Materializer, SystemMaterializer}
import spray.json._

import java.io.ByteArrayInputStream
import java.nio.charset.StandardCharsets.UTF_8
import scala.concurrent.{ExecutionContext, Future}

abstract class HttpPluginRepository(name: String, config: PluginRepositoryConfig, cache: PluginMetadataStorage)
  extends CachedPluginRepository[ArtifactResult](_.extensions.values.headOption.map(_.toPluginId(name)))
    with MetadataEntry.Protocol
    with Logging {

  val customConfig: Config = ConfigFactory.parseString("pekko.coordinated-shutdown.run-by-actor-system-terminate=off")
  val fullConfig: Config = ConfigFactory.load(customConfig)
  implicit val system: ActorSystem = ActorSystem(s"xl-plugin-manager_$name", fullConfig)
  implicit val materializer: Materializer = SystemMaterializer.get(system).materializer
  implicit val ec: ExecutionContext = system.dispatcher

  def shutdown(): Future[Unit] = {
    for {
      _ <- system.terminate()
    } yield ()
  }

  override def getFromRemote(pluginId: PluginId.Artifact): Future[ByteArrayInputStream] = {
    logger.info(s"$name: Downloading plugin ${pluginId.id()}")
    for {
      getSha1 <- downloadSha1(pluginId)
      sha1Content <- getSha1
        .onSuccess(_.asByteArray)
        .onFailure((status, msg) => DownloadChecksumException(pluginId, status, msg))

      getPlugin <- downloadArtifact(pluginId)
      pluginContent <- getPlugin
        .onSuccess(_.asByteArray)
        .onFailure((status, msg) => DownloadArtifactException(pluginId, status, msg))

      (sha1, length) <- verifyPlugin(pluginId, sha1Content, pluginContent, getPlugin.entity.contentLengthOption)
    } yield {
      logger.info(s"$name: Downloaded plugin from ${config.server.uri}: ${pluginId.id()} ($length bytes) | SHA1: $sha1")
      new ByteArrayInputStream(pluginContent)
    }
  }

  override def fetchMetadata: Future[Map[ArtifactId, ExtendedMetadata]] = {
    logger.info(s"Downloading metadata for plugin repository $name from ${config.server.uri}")
    for {
      jsonId <- searchLatestMetadataArtifact("json", None)

      getSha1 <- downloadSha1(jsonId)
      sha1Content <- getSha1
        .onSuccess(_.asByteArray)
        .onFailure((status, msg) => DownloadChecksumException(jsonId, status, msg))

      resp <- downloadArtifact(jsonId)
      content <- resp
        .onSuccess(_.asByteArray)
        .onFailure((status, msg) => DownloadArtifactException(jsonId, status, msg))

      (sha1, length) <- verifyPlugin(jsonId, sha1Content, content, resp.entity.contentLengthOption)
    } yield {
      val entries = new String(content, UTF_8)
        .parseJson
        .convertTo[Seq[MetadataEntry]]
      logger.debug(s"$name: Downloaded metadata from ${config.server.uri}: ${jsonId.id()} ($length bytes) | SHA1: $sha1 | ${entries.size} entries")
      entries.map(e => e.artifactId.copy(repository = Some(name)) -> e.metadata).toMap
    }
  }

  override def fetchLogos: Future[ByteArrayInputStream] = {
    logger.debug(s"$name: Downloading logos")
    for {
      logoArtifact <- searchLatestMetadataArtifact("zip", Some("data"))

      getSha1 <- downloadSha1(logoArtifact)
      sha1Content <- getSha1
        .onSuccess(_.asByteArray)
        .onFailure((status, msg) => DownloadChecksumException(logoArtifact, status, msg))

      resp <- downloadArtifact(logoArtifact)
      content <- resp
        .onSuccess(_.asByteArray)
        .onFailure((status, msg) => DownloadArtifactException(logoArtifact, status, msg))

      (sha1, length) <- verifyPlugin(logoArtifact, sha1Content, content, resp.entity.contentLengthOption)
    } yield {
      logger.debug(s"$name: Downloaded metadata logos data from ${config.server.uri}: ${logoArtifact.id()} ($length bytes) | SHA1: $sha1")
      new ByteArrayInputStream(content)
    }
  }

  private def downloadArtifact(pluginId: PluginId.Artifact): Future[HttpResponse] =
    config.server.httpRequest(getRequest(pluginId))

  private def downloadSha1(pluginId: PluginId.Artifact): Future[HttpResponse] =
    config.server.httpRequest(getRequest(pluginId.copy(packaging = s"${pluginId.packaging}.sha1")))

  protected def getRequest(pluginId: PluginId.Artifact): HttpRequest

  protected def searchArtifact(artifact: Option[String]): Future[HttpResponse] =
    config.server.httpRequest(searchRequest(searchArtifactQuery(artifact)))

  protected def searchMetadata(packaging: String, classifier: Option[String] = None): Future[HttpResponse] = {
    config.server.httpRequest(searchRequest(searchMetadataQuery(packaging, classifier)))
  }

  protected def searchRequest(query: Uri.Query): HttpRequest

  protected def searchLatestMetadataArtifact(packaging: String, classifier: Option[String]): Future[PluginId.Artifact]

  private def searchArtifactQuery(query: Option[String]): Uri.Query = paramsToQuery {
    searchParams(config.listMethod) + ("q" -> query)
  }

  private def searchMetadataQuery(packaging: String, classifier: Option[String]): Uri.Query = paramsToQuery {
    searchParams(ListMethod.ByGroupId(
      groupId = config.metadataArtifactId.groupId,
      classifier = classifier
    )) ++ Map(
      "a" -> Some(config.metadataArtifactId.artifactId)
    )
  }

  private def searchParams(method: ListMethod): Map[String, Option[String]] = Map("repositoryId" -> Some(config.repositoryId)) ++ {
    method match {
      case g: ByGroupId => Map(
        "g" -> Some(g.groupId),
        "c" -> g.classifier
      )
    }
  }
}
