package com.xebialabs.plugin.manager.metadata

import org.apache.pekko.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import spray.json._

case class VersionExpr(op: VersionExpr.Op, version: Version)

object VersionExpr {

  sealed trait Op

  object Op {
    case object GEQ extends Op
    case object GT extends Op
    case object LT extends Op
    case object LEQ extends Op
    case object EQ extends Op

    def fromString(op: String): Option[Op] = op match {
      case ">=" => Some(GEQ)
      case ">" => Some(GT)
      case "<" => Some(LT)
      case "<=" => Some(LEQ)
      case "==" => Some(EQ)
      case _ => None
    }

    val opString: Op => String = {
      case GEQ => ">="
      case GT => ">"
      case LT => "<"
      case LEQ => "<="
      case EQ => "=="
    }

    trait Protocol extends SprayJsonSupport with DefaultJsonProtocol {
      val opWriter: JsonWriter[Op] = o => opString(o).toJson
      val opReader: JsonReader[Op] = jsonReader[Op] {
        case JsString(op) =>
          fromString(op).getOrElse(deserializationError(s"Unknown operator: '$op'"))
        case unknown =>
          deserializationError(s"Cannot parse operator: '$unknown'")
      }
      implicit val opFormat: JsonFormat[Op] = jsonFormat(opReader, opWriter)
    }
  }

  def fromString(versionExpr: String): Option[VersionExpr] = {
    val parts = versionExpr.split(' ')
    if (parts.length != 2) {
      None
    } else {
      for {
        op <- Op.fromString(parts(0).trim)
        version <- Version.fromString(parts(1).trim)
      } yield VersionExpr(op, version)
    }
  }

  trait Protocol extends SprayJsonSupport with DefaultJsonProtocol
    with Version.Protocol
    with Op.Protocol {

    val versionExprWriter: JsonWriter[VersionExpr] = jsonWriter[VersionExpr] {
      case VersionExpr(op, version) => (Op.opString(op) ++ " " ++ version.id).toJson
    }
    val versionExprReader: JsonReader[VersionExpr] = jsonReader[VersionExpr] {
      case JsString(str) =>
        fromString(str).getOrElse(deserializationError(s"Cannot parse version expression: '$str'"))
      case unknown =>
        deserializationError(s"Cannot parse version expression: '$unknown'")
    }
    implicit val versionExprFormat: JsonFormat[VersionExpr] = jsonFormat(versionExprReader, versionExprWriter)
  }
}

