package com.xebialabs.plugin.manager.model

import com.xebialabs.plugin.manager.Plugin
import com.xebialabs.plugin.manager.PluginId.{Artifact, LocalFile}
import com.xebialabs.plugin.manager.compatibility.Jdk17Compatibility
import com.xebialabs.plugin.manager.rest.api.PluginSource.LOCAL
import com.xebialabs.plugin.manager.rest.api.PluginStatus.READY_FOR_INSTALL
import com.xebialabs.plugin.manager.rest.api.{PluginSource, PluginStatus}
import com.xebialabs.plugin.manager.util.ChecksumUtil.calculateChecksum
import com.xebialabs.plugin.manager.util.VersionComparisonUtil

import java.nio.file.{Files, Path}
import scala.util.Try

case class DbPlugin(id: Long,
                    name: String,
                    version: Option[String],
                    extension: String,
                    groupId: String,
                    installationStatus: PluginStatus.Value,
                    source: PluginSource.Value,
                    checksum: String,
                    pluginBytesId: Option[Long],
                    bytes: Option[Array[Byte]])

object DbPlugin {

  def apply(plugin: Plugin): DbPlugin = {
    plugin.id match {
      case artifact: Artifact => fromArtifact(artifact, plugin.bytes)
      case localFile: LocalFile => fromLocalFile(localFile, plugin.bytes)
      case _ => throw new IllegalArgumentException("Not a proper plugin for conversion to a dbPlugin")
    }
  }

  def fromLocalFile(localFile: LocalFile, bytes: Array[Byte]): DbPlugin = {
    new DbPlugin(0, localFile.name, Some(localFile.version.map(v => v.id).orNull), localFile.extension, LOCAL.toString, READY_FOR_INSTALL, LOCAL, calculateChecksum(bytes), None, Some(bytes))
  }

  def fromArtifact(artifact: Artifact, bytes: Array[Byte]): DbPlugin = {
    new DbPlugin(0, artifact.artifactId, artifact.version.map(_.id), artifact.packaging, artifact.groupId, READY_FOR_INSTALL, PluginSource.withName(artifact.repository), calculateChecksum(bytes), None, Some(bytes))
  }

  implicit class DbPluginOps(val dbPlugin: DbPlugin) extends AnyVal {
    def isLowerVersionThan(filePlugin: FilePlugin): Boolean = VersionComparisonUtil.isLowerVersion(dbPlugin.version.getOrElse("0"), filePlugin.version.getOrElse("0"))

    def isSameVersionLike(filePlugin: FilePlugin): Boolean = VersionComparisonUtil.isSameVersion(dbPlugin.version.getOrElse("0"), filePlugin.version.getOrElse("0"))

    def higherVersionExistsIn(fsPlugins: Seq[FilePlugin]): Boolean = fsPlugins.exists(fsPlugin => dbPlugin.name.equals(fsPlugin.name) && dbPlugin.isLowerVersionThan(fsPlugin))

    def doesntExistIn(fsPlugins: Seq[FilePlugin]): Boolean = fsPlugins.forall(fsPlugin => !fsPlugin.name.equals(dbPlugin.name))

    def hasDifferentContentOrVersionThanANameMatchIn(fsPlugins: Seq[FilePlugin]): Boolean = fsPlugins.exists(fsPlugin => dbPlugin.name.equals(fsPlugin.name) && (!dbPlugin.isSameVersionLike(fsPlugin) || isContentDifferent(fsPlugin.filePath, dbPlugin.checksum)))

    def isContentDifferent(filePluginPath: Path, dbPluginChecksum: String): Boolean = {
      val filePluginChecksum = calculateChecksum(Files.readAllBytes(filePluginPath))
      !filePluginChecksum.equals(dbPluginChecksum)
    }

    def isJdk17Compatible: Try[Boolean] = Try(Jdk17Compatibility.isCompatible(dbPlugin))
  }

}
