package com.xebialabs.plugin.manager.util

import com.xebialabs.plugin.manager.config.ConfigWrapper
import com.xebialabs.plugin.manager.exception.{InvalidPluginVersionPropertiesException, MissingPluginVersionPropertiesException}
import com.xebialabs.plugin.manager.model.FilePlugin
import com.xebialabs.plugin.manager.rest.api.PluginSource
import com.xebialabs.plugin.manager.rest.api.PluginSource.{LOCAL, XLD_OFFICIAL, XLR_OFFICIAL}
import com.xebialabs.xlplatform.sugar.PathSugar.path2File
import grizzled.slf4j.Logging

import java.io.File
import java.nio.file.{Files, Path}
import scala.jdk.CollectionConverters.ListHasAsScala
import scala.util.{Failure, Success, Using}

object PluginsTraverser extends Logging {

  private def listPlugins(dir: File): List[FilePlugin] = {
    dir.listFiles()
      .filter(file => ConfigWrapper.isValidExtension(file.toString))
      .map(filePath => FilePlugin(filePath.toPath))
      .toList
  }

  def traverse(dir: Path)(officialCallback: (PluginSource.Value, List[FilePlugin]) => Unit, localCallback: List[FilePlugin] => Unit): Unit = {
    Using(Files.list(dir)) { stream =>
      stream.filter(_.isDirectory)
        .map(dir => PluginSource.withName(dir.getFileName.toString) -> listPlugins(dir))
        .forEach(fsPluginsBySourceEntry => {
          fsPluginsBySourceEntry._1 match {
            case XLD_OFFICIAL | XLR_OFFICIAL =>
              officialCallback(fsPluginsBySourceEntry._1, fsPluginsBySourceEntry._2)
            case LOCAL =>
              localCallback(fsPluginsBySourceEntry._2)
            case _ => error(s"Could not match plugins directory(source) value ${fsPluginsBySourceEntry._1.toString}, skipping..")
          }
        })
    } match {
      case Success(_) =>
        info(s"Successfully synced plugins")
      case Failure(err) =>
        error(s"Error during plugin sync with message: ${err.getMessage}")
        err match {
          case _: InvalidPluginVersionPropertiesException | _: MissingPluginVersionPropertiesException => throw err
          case _ => ()
        }
    }
  }

  def traverseAndReturnOfficial(dir: Path)(officialCallback: (PluginSource.Value, List[FilePlugin]) => List[FilePlugin], localCallback: List[FilePlugin] => Unit): List[FilePlugin] = {
    Using(Files.list(dir)) { stream =>
      stream.toList.asScala.filter(_.isDirectory)
        .map(dir => PluginSource.withName(dir.getFileName.toString) -> listPlugins(dir))
        .flatMap(fsPluginsBySourceEntry => fsPluginsBySourceEntry._1 match {
          case XLD_OFFICIAL | XLR_OFFICIAL =>
            officialCallback(fsPluginsBySourceEntry._1, fsPluginsBySourceEntry._2)
          case LOCAL => localCallback(fsPluginsBySourceEntry._2)
            List.empty
          case _ => error(s"Could not match plugins directory(source) value ${fsPluginsBySourceEntry._1.toString}, skipping..")
            List.empty
        }).toList
    } match {
      case Success(plugins) =>
        info(s"Successfully synced plugins")
        plugins
      case Failure(err) =>
        error(s"Error during plugin sync with message: ${err.getMessage}")
        err match {
          case _: InvalidPluginVersionPropertiesException | _: MissingPluginVersionPropertiesException => throw err
          case _ => List.empty
        }
    }
  }
}
