package com.xebialabs.plugin.manager.metadata

import akka.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import akka.http.scaladsl.model.Uri
import spray.json._


case class ExtendedMetadata(name: String,
                            author: String,
                            description: String,
                            categories: Set[String],
                            features: Set[String],
                            links: Map[String, Uri],
                            supportedBy: SupportedBy,
                            maturity: MaturityLevel,
                            requirements: Requirements,
                            screenshots: Map[String, Uri])

object ExtendedMetadata {
  trait Protocol extends SprayJsonSupport with DefaultJsonProtocol
    with ExternalLink.Protocol
    with SupportedBy.Protocol
    with MaturityLevel.Protocol
    with Requirements.Protocol {

    val extendedMetadataReader: JsonReader[ExtendedMetadata] = jsonReader[ExtendedMetadata] {
      case obj: JsObject =>
        obj.getFields(
          "name", "author", "description",
          "categories", "features",
          "links", "supportedBy", "maturity",
          "requirements", "screenshots") match {
          case Seq(
            JsString(name), JsString(author), JsString(description),
            JsArray(categories), JsArray(features),
            JsArray(links), supportedBy, maturity,
            requirements, JsArray(screenshots)
          ) =>
            ExtendedMetadata(
              name = name,
              author = author,
              description = description,
              categories = categories.map(_.convertTo[String]).toSet,
              features = features.map(_.convertTo[String]).toSet,
              links = links.map(_.convertTo[ExternalLink]).map(e => e.name -> e.url).toMap,
              supportedBy = supportedBy.convertTo[SupportedBy],
              maturity = maturity.convertTo[MaturityLevel],
              requirements = requirements.convertTo[Requirements],
              screenshots = screenshots.map(_.convertTo[ExternalLink]).map(s => s.name -> s.url).toMap
            )
          case other =>
            deserializationError(s"Cannot parse metadata: '$other'")
        }
      case unknown =>
        deserializationError(s"Cannot parse metadata: '$unknown'")
    }
    val extendedMetadataWriter: JsonWriter[ExtendedMetadata] = jsonWriter[ExtendedMetadata] { em =>
      em.links.map(t => ExternalLink.apply(t._1, t._2))
      JsObject(
        "name" -> em.name.toJson,
        "author" -> em.author.toJson,
        "description" -> em.description.toJson,
        "categories" -> JsArray(em.categories.map(_.toJson).toSeq: _*),
        "features" -> JsArray(em.features.map(_.toJson).toSeq: _*),
        "links" -> JsArray(em.links.map(t => ExternalLink(t._1, t._2).toJson).toSeq: _*),
        "supportedBy" -> em.supportedBy.toJson,
        "maturity" -> em.maturity.toJson,
        "requirements" -> em.requirements.toJson,
        "screenshots" -> JsArray(em.screenshots.map(t => ExternalLink(t._1, t._2).toJson).toSeq: _*)
      )
    }
    implicit val extendedMetadataFormat: JsonFormat[ExtendedMetadata] = jsonFormat(extendedMetadataReader, extendedMetadataWriter)
  }
}