package com.xebialabs.plugin.manager.validator

import com.xebialabs.plugin.manager.PluginId
import com.xebialabs.plugin.manager.config.ConfigWrapper
import grizzled.slf4j.Logging
import org.apache.commons.io.file.PathUtils

import java.io.File
import java.util.zip.ZipFile
import scala.util.{Failure, Success, Try}

class PluginValidator() extends Logging {
  private val validator = new SyntheticValidatorAtInstallation()

  def validate(bytes: Array[Byte], pluginId: PluginId): ValidationResult = {
    val pluginFile = ValidatorUtils.createFileInWorkFolderFrom(bytes, "validation-" + pluginId.name)

    val zipFile = Try(new ZipFile(pluginFile))
    val result = if (zipFile.isSuccess) {
      zipFile.get.close()
      validateInternal(pluginFile, pluginId)
    } else {
      InvalidArchive(pluginId.name)
    } // in the similar FileUtils implementation there is by default force delete of read only folders, that is not supported on some container envs, so using PathUtils
    PathUtils.deleteDirectory(pluginFile.getParentFile.toPath, PathUtils.EMPTY_LINK_OPTION_ARRAY)
    result
  }

  private def getExtractor(pluginId: PluginId): Try[SyntheticExtractor] = {
    val extension = pluginId match {
      case PluginId.LocalFile(_, _, extension) => extension
      case PluginId.Artifact(_, _, _, _, packaging, _) => packaging
    }

    extension match {
      case ConfigWrapper.EXTENSION_JAR => Success(new JarSyntheticExtractor())
      case ConfigWrapper.EXTENSION_XLDP => Success(new XldpSyntheticExtractor())
      case ConfigWrapper.EXTENSION_ZIP => Success(new ReloadableSyntheticExtractor())
      case _ => Failure(new IllegalArgumentException(s"unsupported plugin extension $extension"))
    }
  }

  private def validateInternal(pluginFile: File, pluginId: PluginId): ValidationResult = {
    val syntheticOpt = getExtractor(pluginId) match {
      case Success(extractor) => extractor.extractFrom(pluginFile, pluginId)
      case Failure(exception) => Some(Failure(exception))
    }
    syntheticOpt match {
      case Some(Success(types)) =>
        validator.validate(types)
      case Some(Failure(exception)) =>
        logger.debug("Type validation failed. Invalid xml or yaml. Message: " + exception.getMessage)
        TypeDefinitionNotValid(pluginId.name)
      case None => PluginValid()
    }
  }
}
