package com.xebialabs.plugin.manager.repository.sql

import org.springframework.jdbc.core.JdbcTemplate

import scala.util.Using

sealed abstract class Dialect(defaultCase: DefaultCase.Value) {
  private val upperRegex = "[A-Z_]*".r
  private val lowerRegex = "[a-z_]*".r

  def withDialect(stringToConvert: String): String
  def quoteIfNecessary(name: String): String = (defaultCase, name) match {
    case (DefaultCase.UPPER, lowerRegex()) => s"\"$name\""
    case (DefaultCase.LOWER, upperRegex()) => s"\"$name\""
    case (_, _) => name
  }

}

class CommonDialect(defaultCase: DefaultCase.Value = DefaultCase.NONE) extends Dialect(defaultCase = defaultCase) {
  override def withDialect(stringToConvert: String): String = stringToConvert.toUpperCase
}

case class PostgresDialect() extends CommonDialect(defaultCase = DefaultCase.LOWER) {
  override def withDialect(stringToConvert: String): String = stringToConvert.toLowerCase
}

case class MsSqlDialect() extends CommonDialect
case class MySqlDialect() extends CommonDialect
case class Db2Dialect() extends CommonDialect(defaultCase = DefaultCase.UPPER)
case class OracleDialect() extends CommonDialect(defaultCase = DefaultCase.UPPER)
case class DerbyDialect() extends CommonDialect(defaultCase = DefaultCase.UPPER)

object DefaultCase extends Enumeration {
  val UPPER, LOWER, NONE = Value
}

object Dialect {
  val Oracle = """(oracle)""".r.unanchored
  val MySQL = """(mysql)""".r.unanchored
  val PostgreSQL = """(postgresql)""".r.unanchored
  val DB2 = """(db2)""".r.unanchored
  val MSSQL = """(sql server)""".r.unanchored
  val H2 = """(h2)""".r.unanchored
  val Derby = """(derby)""".r.unanchored

  def apply(jdbcTemplate: JdbcTemplate): Dialect = {
    Using(jdbcTemplate.getDataSource.getConnection()) { connection =>
      val productName: String = connection.getMetaData.getDatabaseProductName.toLowerCase
      productName match {
        case PostgreSQL(_) => new PostgresDialect
        case MSSQL(_) => new MsSqlDialect
        case MySQL(_) => new MySqlDialect
        case DB2(_) => new Db2Dialect
        case Oracle(_) => new OracleDialect
        case Derby(_) => new DerbyDialect
        case H2(_) => new CommonDialect
        case _ => throw new IllegalArgumentException(s"Could not determine the correct dialect for product name: $productName")
      }
    }.get
  }
}