package com.xebialabs.plugin.manager.metadata

import akka.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import com.xebialabs.plugin.manager.{LocalPluginManager, PluginId, requireProperFilename}
import spray.json._

case class ArtifactId(groupId: String, artifactId: String, version: Version, packaging: String, repository: Option[String] = None) {
  requireProperFilename(groupId, "groupId")
  requireProperFilename(artifactId, "artifactId")
  repository.foreach(r => {
    requireProperFilename(r, "repository")
  })
}

object ArtifactId {
  implicit class ArtifactIdOps(val artifactId: ArtifactId) extends AnyVal {
    def id: String =
      s"${artifactId.repository.getOrElse(LocalPluginManager.name)}:${artifactId.groupId}:${artifactId.artifactId}-${artifactId.version.id}"

    def toPluginId(classifier: Option[String] = None): PluginId = artifactId.repository match {
      case None if artifactId.groupId == PluginId.localGroupId =>
        toLocal
      case Some(repo) if artifactId.groupId != PluginId.localGroupId && repo != LocalPluginManager.name =>
        toArtifact(repo, classifier)
      case _ => throw new IllegalArgumentException(s"Can't convert artifacts in repository ${artifactId.repository} to a plugin id")
    }

    def toArtifact(repositoryName: String, classifier: Option[String] = None): PluginId.Artifact =
      PluginId.Artifact.parsed(repositoryName, artifactId.groupId, artifactId.artifactId, artifactId.version, artifactId.packaging, classifier)

    def toArtifact: PluginId.Artifact =
      PluginId.Artifact.parsed(artifactId.repository.getOrElse(PluginId.localGroupId), artifactId.groupId, artifactId.artifactId, artifactId.version, artifactId.packaging,  None)

    def toLocal: PluginId.LocalFile =
      PluginId.LocalFile.parsed(artifactId.artifactId, Some(artifactId.version), artifactId.packaging)
  }

  trait Protocol extends SprayJsonSupport with DefaultJsonProtocol
    with Version.Protocol {

    val artifactIdReader: RootJsonReader[ArtifactId] = {
      case obj: JsObject =>
        obj.getFields("groupId", "artifactId", "version", "packaging") match {
          case Seq(JsString(groupId), JsString(artifactId), version, JsString(packaging)) =>
            ArtifactId(groupId, artifactId, version.convertTo[Version], packaging)
          case wrong =>
            deserializationError(s"Cannot parse artifactId object: $wrong")
        }
      case unknown =>
        deserializationError(s"Cannot parse artifactId object: $unknown")
    }

    val artifactIdWriter: RootJsonWriter[ArtifactId] = (id: ArtifactId) => JsObject(
      "groupId" -> id.groupId.toJson,
      "artifactId" -> id.artifactId.toJson,
      "version" -> id.version.toJson,
      "packaging" -> id.packaging.toJson
    )

    implicit val artifactIdFormat: RootJsonFormat[ArtifactId] = rootJsonFormat(artifactIdReader, artifactIdWriter)
  }
}
