package com.xebialabs.plugin.manager.repository.storage

import java.io.{File, InputStream}
import java.nio.file.{Path, Paths}

import com.xebialabs.plugin.manager.metadata._
import com.xebialabs.plugin.manager.{LocalPluginManager, PluginId}
import com.xebialabs.xlplatform.io.ZipUtils
import grizzled.slf4j.Logging
import org.apache.commons.io.FileUtils

import scala.collection.mutable

// Note: not purely "MemoryStorage" anymore, since we persist the content of the zip file...
class PluginMetadataMemoryStorage(val repositoryName: String) extends PluginMetadataStorage with Logging {
  protected val cacheDir: Path = Paths.get("cache").resolve("plugins")

  protected val memory: mutable.Map[ArtifactId, PluginMetadata] = mutable.Map.empty

  override def clear(): Unit = this.synchronized {
    memory.clear()
  }

  // TODO: validate content of zipfile!
  override def storeLogos(artifactIds: Seq[String])(data: InputStream): Seq[String] = this.synchronized {
    val dir = cacheDir.resolve(repositoryName).toFile
    FileUtils.deleteDirectory(dir)
    ZipUtils.collect(data, dir) {
      case filename if artifactIds contains filename.dropRight(4) =>
        filename.dropRight(4)
    }
  }

  override def store(data: Seq[PluginId.Artifact], metadata: Map[ArtifactId, ExtendedMetadata]): Unit = this.synchronized {
    memory.clear()
    memory ++= merge(data, metadata)
  }

  override def get(id: ArtifactId): Option[PluginMetadata] =
    memory.get(id)

  override def getLogo(artifactId: String): Option[File] = this.synchronized {
    val dir = cacheDir.resolve(repositoryName)
    logger.debug(s"search logo data for $artifactId in $dir")
    if (dir.toFile.isDirectory) {
      LocalPluginManager
        .listDir(dir)
        .find(_.getFileName.toString.startsWith(s"$artifactId."))
        .map(_.toFile)
    } else {
      logger.debug(s"not a directory: ${dir.toString}")
      None
    }
  }

  override def search(query: Option[String]): Map[ArtifactId, Option[PluginMetadata]] = {
    query match {
      case Some(name) if name.nonEmpty =>
        readonly
          .view
          .filterKeys(_.toPluginId().name contains name)
          .toMap

      case _ =>
        readonly
    }
  }

  protected def merge(data: Seq[PluginId.Artifact],
                      metadata: Map[ArtifactId, ExtendedMetadata]): Seq[(ArtifactId, PluginMetadata)] = {
    for {
      (artifactId, group) <- data.groupBy(_.toArtifactId)
      if group.nonEmpty
      meta <- metadata.get(artifactId)
      hasLogo = getLogo(artifactId.artifactId).isDefined
    } yield {
      artifactId -> PluginMetadata(meta, hasLogo)
    }
  }.toSeq

  protected def readonly: Map[ArtifactId, Option[PluginMetadata]] = memory.view.mapValues(Some.apply).toMap

}
