package com.xebialabs.plugin.manager.repository.nexus

import akka.actor.ActorSystem
import akka.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import akka.http.scaladsl.model.headers.{BasicHttpCredentials, HttpCredentials}
import akka.http.scaladsl.model.{HttpRequest, HttpResponse, Uri}
import akka.http.scaladsl.settings.ConnectionPoolSettings
import akka.http.scaladsl.{ClientTransport, Http}
import com.xebialabs.plugin.manager.config.{Server, Credentials, Proxy}
import spray.json._

import java.net.InetSocketAddress
import scala.concurrent.Future
import scala.util.{Failure, Success, Try}

case class NexusServerConfig(uri: Uri, credentials: HttpCredentials, transport: ClientTransport = ClientTransport.TCP) {
  def serviceUri: Uri = uri.withPath(uri.path / "service" / "local")

  def settings(implicit system: ActorSystem): ConnectionPoolSettings = ConnectionPoolSettings(system).withTransport(transport)

  def httpRequest(req: HttpRequest)(implicit system: ActorSystem): Future[HttpResponse] =
    Http().singleRequest(req, settings = settings)
}

object NexusServerConfig {
  def apply(uri: Uri, credentials: HttpCredentials, proxyUri: Uri, proxyCredentialsOpt: Option[HttpCredentials]): NexusServerConfig = {
    val proxyAddress = new InetSocketAddress(proxyUri.authority.host.address(), proxyUri.authority.port)
    val transport = proxyCredentialsOpt match {
      case None => ClientTransport.httpsProxy(proxyAddress)
      case Some(proxyCredentials) => ClientTransport.httpsProxy(proxyAddress, proxyCredentials)
    }
    NexusServerConfig(uri, credentials, transport)
  }

  def fromConfig(serverConfig: Server): Try[NexusServerConfig] = {
    // validations first
    if (!"nexus".equals(serverConfig.getServerType))
      return Failure(new IllegalArgumentException(s"Plugin server ${serverConfig.getName} must have a 'nexus' server type"))

    if (serverConfig.getUrl == null || Try(Uri.apply(serverConfig.getUrl)).isFailure)
      return Failure(new IllegalArgumentException(s"Invalid url for plugin server ${serverConfig.getName}"))

    val credentials = serverConfig.getCredentials
    if (credentials == null || credentials.getPassword == null || credentials.getUsername == null)
      return Failure(new IllegalArgumentException(s"Both username and password must be present in the '${serverConfig.getName}' plugin server configuration"))

    if (serverConfig.getProxy != null && !isValid(serverConfig.getProxy))
      return Failure(new IllegalArgumentException(s"Invalid proxy configured for plugin server '${serverConfig.getName}''"))

    // creating a server config
    val httpCredentials = new BasicHttpCredentials(credentials.getUsername, credentials.getPassword)
    val uri = Uri.apply(serverConfig.getUrl)
    getProxyTransport(serverConfig) match {
      case Some(proxy) => Success(new NexusServerConfig(uri, httpCredentials, proxy))
      case None => Success(new NexusServerConfig(uri, httpCredentials))
    }
  }

  def getProxyTransport(serverConfig: Server): Option[ClientTransport] = {
    if (serverConfig.getProxy != null) {
      val proxy = serverConfig.getProxy
      val address = new InetSocketAddress(proxy.getHost, proxy.getPort)
      if (proxy.getCredentials != null && hasValidCredentials(proxy.getCredentials)) {
        val credentials = serverConfig.getProxy.getCredentials
        Some(ClientTransport.httpsProxy(address, new BasicHttpCredentials(credentials.getUsername, credentials.getPassword)))
      } else {
        Some(ClientTransport.httpsProxy(address))
      }
    } else {
      None
    }
  }

  def isValid(proxy: Proxy): Boolean = {
    val validCredentials = proxy.getCredentials == null || proxy.getCredentials.getUsername != null && proxy.getCredentials.getPassword != null
    val validAddress = proxy.getHost != null && proxy.getPort != null
    validCredentials && validAddress
  }

  def hasValidCredentials(credentials: Credentials): Boolean = credentials.getPassword != null && credentials.getUsername != null

  trait Protocol extends SprayJsonSupport with DefaultJsonProtocol {
    implicit val nexusServerConfigWriter: RootJsonWriter[NexusServerConfig] = {
      case NexusServerConfig(uri, _, _) =>
        JsObject(
          "server-type" -> "nexus".toJson,
          "url" -> uri.toString.toJson
        )
    }
  }
}

