package com.xebialabs.plugin.manager.startup

import com.xebialabs.plugin.manager.config.ConfigWrapper
import com.xebialabs.plugin.manager.metadata.XLProduct
import com.xebialabs.plugin.manager.model.FilePlugin
import com.xebialabs.plugin.manager.repository.sql.SqlPluginRepository
import com.xebialabs.plugin.manager.rest.api.PluginStatus
import com.xebialabs.plugin.manager.util.PluginsTraverser.traverse
import grizzled.slf4j.Logging

import java.nio.file.{Path, Paths}

/**
  * Inserts all plugins from official and local folder into the DB.
  *
  * Intended to be run on fresh product installation.
  */
class PluginInserter(val pluginRepository: SqlPluginRepository, val product: String, val pluginsDir: Path) extends Logging {

  def this(pluginRepository: SqlPluginRepository, product: String) = {
    this(pluginRepository, product, Paths.get("plugins"))
  }

  val xlProduct: XLProduct = XLProduct.fromString(product)

  def insertAllPluginsFromFsIntoDb(): Unit = {
    traverse(pluginsDir, ConfigWrapper.extension)(
      (_, plugins) => insertOfficial(plugins),
      plugins => insertLocal(plugins)
    )

    pluginRepository.updateAllPluginsStatusTo(PluginStatus.INSTALLED)
  }

  private def insertOfficial(fsPlugins: List[FilePlugin]): Unit = {
    debug(s"Processing list of official plugins found on the filesystem $fsPlugins")
    fsPlugins.foreach(fsPlugin => writeOfficialToDatabase(fsPlugin))
  }

  private def insertLocal(plugins: List[FilePlugin]): Unit = {
    debug(s"Processing list of local plugins found on the filesystem $plugins")
    plugins.foreach(fsPlugin => writeLocalToDatabase(fsPlugin))
  }

  private def writeOfficialToDatabase(plugin: FilePlugin): Unit = {
    info(s"Inserting plugin ${plugin.name} from filesystem to database")
    pluginRepository.insert(plugin.toOfficialDbPlugin(xlProduct))
  }

  private def writeLocalToDatabase(plugin: FilePlugin): Unit = {
    info(s"Inserting plugin ${plugin.name} from filesystem to database")
    pluginRepository.insert(plugin.toLocalDbPlugin)
  }

}
