package com.xebialabs.plugin.manager.cli

import com.xebialabs.plugin.manager.LocalPluginManagerFactory
import com.xebialabs.plugin.manager.PluginId.LocalFile
import com.xebialabs.plugin.manager.cli.PluginManagerCliLaunchOptions.{UtilListOps, getPluginPathNormalized}
import com.xebialabs.plugin.manager.config.ConfigWrapper
import com.xebialabs.plugin.manager.config.impl.PluginManagerPropertiesImpl
import com.xebialabs.plugin.manager.metadata.XLProduct
import com.xebialabs.plugin.manager.metadata.XLProduct.id
import com.xebialabs.plugin.manager.model.{DbPlugin, FilePlugin}
import com.xebialabs.plugin.manager.repository.sql.SqlPluginRepository
import com.xebialabs.plugin.manager.service.{LocalPluginService, PluginService}
import com.xebialabs.xlplatform.sugar.PathSugar.path2File
import org.apache.commons.io.FileUtils

import java.io.File
import java.nio.file.{Files, Path, Paths, StandardCopyOption}
import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters.IteratorHasAsScala
import scala.util.matching.Regex

class PluginManagerCli(pluginRepository: SqlPluginRepository, product: XLProduct, version: String) {

  private val pluginService: PluginService = new LocalPluginService(new PluginManagerPropertiesImpl, id(product), version, pluginRepository)
  private val output: StringBuffer = new StringBuffer(System.lineSeparator())

  private def println(s: String): StringBuffer = output.append(s).append(System.lineSeparator())

  def run(options: PluginManagerCliLaunchOptions): StringBuffer = {
    ConfigWrapper.initWith(product)

    val pluginsFromDb: Seq[DbPlugin] = pluginRepository.getAllWithoutBytes
    val pluginsFromFs: Seq[FilePlugin] = getFromFilesystem
    val context = new CommandContext(pluginsFromDb, pluginsFromFs)

    val optionalCommand = createCommand(options)
    optionalCommand.foreach(_.executeIn(context))

    output
  }

  class CommandContext(var pluginsFromDb: Seq[DbPlugin], var pluginsFromFs: Seq[FilePlugin]) {
    def isAcceptingCommands: Boolean = pluginsFromDb.nonEmpty || pluginsFromFs.nonEmpty
  }

  private def createCommand(options: PluginManagerCliLaunchOptions): Option[Command] = {
    if (Option(options.add).isDefined) Some(AddCommand(options.add))
    else if (Option(options.update).isDefined) Some(UpdateCommand(options))
    else if (Option(options.delete).isDefined) Some(DeleteCommand(options))
    else if (Option(options.list).isDefined) Some(ListCommand())
    else None
  }

  private def getFromFilesystem: Seq[FilePlugin] = {
    val pluginsDir: Path = LocalPluginManagerFactory.pluginsPath
    if (Files.exists(pluginsDir)) {
      Files.walk(pluginsDir)
        .filter(_.isFile)
        .filter(file => file.getFileName.toString.endsWith(ConfigWrapper.extension))
        .map(filePath => FilePlugin(filePath))
        .iterator().asScala.toSeq
    } else {
      Seq.empty
    }
  }

  def printPlugins(dbPlugins: Seq[DbPlugin], fsPlugins: Seq[FilePlugin]): StringBuffer = {
    println(String.format("%-40s%-20s%-10s%-10s", "NAME", "VERSION", "DATABASE", "FILESYSTEM"))
    println("-------------------------------------------------------------------------------------------")

    val printablePlugins: List[PrintablePlugin] = getPrintablePlugins(dbPlugins, fsPlugins)
    if (printablePlugins.nonEmpty) {
      printablePlugins.sortWith(_.name < _.name).foreach(printablePlugin => {
        val isDbPlugin = if (printablePlugin.dbPlugin) "+" else "-"
        val isFsPlugin = if (printablePlugin.fsPlugin) "+" else "-"
        println(String.format("%-40s%-24s%-12s%-16s", printablePlugin.name, printablePlugin.version.getOrElse(""), isDbPlugin, isFsPlugin))
      })
    } else println("No plugins in database or filesystem")

    println("-------------------------------------------------------------------------------------------")
  }

  case class PrintablePlugin(
                              name: String,
                              version: Option[String],
                              dbPlugin: Boolean,
                              fsPlugin: Boolean
                            )

  def getPrintablePlugins(dbPlugins: Seq[DbPlugin], fsPlugins: Seq[FilePlugin]): List[PrintablePlugin] = {
    val printablePlugins = new ListBuffer[PrintablePlugin]
    if (dbPlugins.nonEmpty) {
      dbPlugins.foreach(dbPlugin => {
        val existInFs = fsPlugins.exists(fsPlugin => fsPlugin.name.equals(dbPlugin.name) && dbPlugin.version.getOrElse("").equals(fsPlugin.version.getOrElse("")))
        printablePlugins += PrintablePlugin(dbPlugin.name, dbPlugin.version, dbPlugin = true, fsPlugin = existInFs)
      })
    }
    if (fsPlugins.nonEmpty) {
      fsPlugins.foreach(fsPlugin => {
        if (!printablePlugins.exists(printablePlugin => printablePlugin.name.equals(fsPlugin.name) && printablePlugin.version.getOrElse("").equals(fsPlugin.version.getOrElse("")))) {
          printablePlugins += PrintablePlugin(fsPlugin.name, fsPlugin.version, dbPlugin = false, fsPlugin = true)
        }
      })
    }
    printablePlugins.toList
  }

  trait Command {
    def executeIn(commandContext: CommandContext): Unit
  }

  case class ListCommand() extends Command {
    def executeIn(commandContext: CommandContext): Unit = printPlugins(commandContext.pluginsFromDb, commandContext.pluginsFromFs)
  }

  object ListCommand {
    val regex = "list"
  }

  case class AddCommand(pluginToAdd: String) extends Command {
    override def executeIn(commandContext: CommandContext): Unit = {
      val pluginPath = getPluginPathNormalized(Paths.get(pluginToAdd))
      if (pluginPath.isDefined) {
        val pluginName = pluginPath.get.getFileName.toString
        val localFile = LocalFile(pluginName)
        if (isInPluginFolder(pluginPath.get)) {
          val workFolder = Paths.get("work" + File.separator + "cli")
          workFolder.mkdirs()

          val pluginFileInWorkFolder = Paths.get(workFolder.getPath, pluginName)
          Files.move(pluginPath.get, pluginFileInWorkFolder, StandardCopyOption.REPLACE_EXISTING)

          try {
            val bytes = Files.readAllBytes(pluginFileInWorkFolder)
            val result = pluginService.install(localFile, bytes)
            println(result.message)
          } catch {
            case e: Exception =>
              println("Error while installing plugin: " + e.getMessage)
          }

          Files.move(pluginFileInWorkFolder, pluginPath.get, StandardCopyOption.REPLACE_EXISTING)
          workFolder.delete()
        } else {
          val bytes = Files.readAllBytes(pluginPath.get)
          val result = pluginService.install(localFile, bytes)
          println(result.message)
        }
      } else {
        println(s"File $pluginToAdd doesn't exist")
      }
    }
  }

  object AddCommand {
    val regex = "add"
  }

  case class DeleteCommand(options: PluginManagerCliLaunchOptions) extends Command {
    def executeIn(context: CommandContext): Unit = {
      val pluginToDelete = options.delete.get(0)
      val version = if (options.delete.size() > 1) Option(options.delete.get(1)) else None

      val pluginInDb = context.pluginsFromDb.filter(dbPlugin => dbPlugin.name.equals(pluginToDelete) && (version.isEmpty || dbPlugin.version == version))
      val pluginInFs = context.pluginsFromFs.filter(fsPlugin => fsPlugin.name.equals(pluginToDelete) && (version.isEmpty || fsPlugin.version == version))

      if (pluginInDb.isEmpty && pluginInFs.isEmpty) {
        println("Plugin not found.")
      } else if (pluginInDb.size > 1 || pluginInFs.size > 1) {
        println("There are multiple plugins found. Please specify version.")
      } else {
        delete(context, pluginInDb.nonEmpty, pluginInFs.nonEmpty, pluginToDelete, version)
      }
    }

    def delete(context: CommandContext, shouldDeleteFromDb: Boolean, shouldDeleteFromFs: Boolean, pluginToDelete: String, version: Option[String]): Unit = {
      if (shouldDeleteFromFs) {
        deleteFromFs(pluginToDelete, version, context.pluginsFromFs)
      }

      if (shouldDeleteFromDb) {
        deleteFromDb(pluginToDelete, version, context.pluginsFromDb)
      }

      println("Please verify and delete plugin file in other cluster members' plugins directory if needed")
    }

    def deleteFromFs(pluginToDelete: String, version: Option[String], plugins: Seq[FilePlugin]): Unit = {
      FileUtils.forceDelete(plugins.filter(plugin => plugin.name == pluginToDelete && (version.isEmpty || plugin.version == version)).head.filePath)
      println(s"$pluginToDelete deleted from filesystem")
    }

    def deleteFromDb(pluginToDelete: String, version: Option[String], plugins: Seq[DbPlugin]): Unit = {
      pluginRepository.delete(plugins.filter(plugin => plugin.name == pluginToDelete && (version.isEmpty || plugin.version == version)).head)
      println(s"$pluginToDelete deleted from database")
    }
  }

  object DeleteCommand {
    val regex: Regex = "delete (.+)".r
  }

  case class UpdateCommand(options: PluginManagerCliLaunchOptions) extends Command {

    def executeIn(context: CommandContext): Unit = {
      options.update.validate match {
        case WrongUsage() => println("Wrong usage of update command. There have to be exactly 2 arguments.")
        case PluginAtPathDoesntExist(pluginToAdd) => println(s"File $pluginToAdd doesn't exist")
        case Valid(pluginName, pluginPath) =>
          val pluginNameWithVersion = pluginPath.getFileName.toString
          val localFile = LocalFile(pluginNameWithVersion)

          if (isInPluginFolder(pluginPath)) {
            val pluginFileInWorkFolder = movePluginToWorkFolder(pluginName, pluginPath)

            doUpdate(pluginFileInWorkFolder, pluginName, localFile)

            moveBackToOriginalLocationAndDeleteWorkFolder(pluginFileInWorkFolder, pluginPath)
          } else {
            doUpdate(pluginPath, pluginName, localFile)
          }
      }
    }

    private def doUpdate(pluginPath: Path, pluginName: String, localFile: LocalFile) = {
      val bytes = Files.readAllBytes(pluginPath)
      val result = pluginService.update(pluginName, localFile, bytes)
      println(result.message)
    }

    private def deleteWorkFolder() = Paths.get("work" + File.separator + "cli").delete()

    private def moveBackToOriginalLocationAndDeleteWorkFolder(pluginFileInWorkFolder: Path, pluginPath: Path) = {
      Files.move(pluginFileInWorkFolder, pluginPath, StandardCopyOption.REPLACE_EXISTING)
      deleteWorkFolder()
    }

    private def movePluginToWorkFolder(pluginName: String, pluginPath: Path) = {
      val workFolder = Paths.get("work" + File.separator + "cli")
      workFolder.mkdirs()

      val pluginFileInWorkFolder = Paths.get(workFolder.getPath, pluginName)
      Files.move(pluginPath, pluginFileInWorkFolder, StandardCopyOption.REPLACE_EXISTING)
      pluginFileInWorkFolder
    }
  }

  object UpdateCommand {
    val regex: Regex = "update (.+)".r
  }

  private def isInPluginFolder(pluginToAdd: Path): Boolean = {
    if (Files.exists(LocalPluginManagerFactory.pluginsPath)) {
      val pluginFile = Files.walk(LocalPluginManagerFactory.pluginsPath).filter(Files.isSameFile(_, pluginToAdd)).findAny()
      pluginFile.isPresent
    } else {
      false
    }
  }
}
