package com.xebialabs.plugin.manager.service

import com.xebialabs.deployit.engine.spi.exception.{DeployitException, HttpResponseCodeResult}
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.plugin.manager.PluginId.{Artifact, LocalFile}
import com.xebialabs.plugin.manager.config.PluginManagerProperties
import com.xebialabs.plugin.manager.metadata.{ArtifactId, PluginMetadata}
import com.xebialabs.plugin.manager.model.DbPlugin
import com.xebialabs.plugin.manager.repository.PluginsRepository
import com.xebialabs.plugin.manager.repository.nexus.{NexusPluginRepository, NexusRepositoryConfig, NexusServerConfig}
import com.xebialabs.plugin.manager.rest.api.{PluginSource, PluginStatus}
import com.xebialabs.plugin.manager.rest.dto.PluginDto
import com.xebialabs.plugin.manager.service.PluginService.defaultTimeout
import com.xebialabs.plugin.manager.validator.PluginValidatorAtInstallation
import com.xebialabs.plugin.manager.{Plugin, PluginId, PluginManager}
import grizzled.slf4j.Logging

import java.io.File
import javax.annotation.PreDestroy
import scala.collection.mutable
import scala.concurrent.duration._
import scala.concurrent.{Await, ExecutionContext, Future}
import scala.jdk.CollectionConverters._
import scala.language.postfixOps
import scala.util.{Failure, Success, Try}

trait PluginService extends Logging {

  def pluginManager: PluginManager

  implicit lazy val ec: ExecutionContext = pluginManager.ec

  def repositories: mutable.Map[String, PluginsRepository]

  def addRepository(repository: PluginsRepository): Boolean

  def deleteRepository(name: String): Boolean


  def update(): Unit = {
    if (repositories.nonEmpty) {
      Await.ready(Future.sequence(repositories.values.map(_.update())), defaultTimeout)
    }
  }

  def search(query: Option[String]): Map[ArtifactId, Option[PluginMetadata]] = {
    logger.debug(s"search($query)")
    Await.result(
      for {
        installed <- pluginManager.search(query)
        available <- Future.sequence(repositories.values.map(_.search(query))).map(_.flatten)
      } yield {
        available.foldLeft(installed.map(_ -> Option.empty[PluginMetadata]).toMap) {
          case (acc, (id, pm1)) =>
            acc.get(id) match {
              case None =>
                acc.updated(id, pm1)
              case Some(_) =>
                pm1.map(pm => acc.updated(id, Some(pm))).getOrElse(acc)
            }
        }
      },
      defaultTimeout
    )
  }

  def searchOfficial(query: String): Map[ArtifactId, Option[PluginMetadata]] = {
    Await.result(
      for {
        available <- Future.sequence(repositories.values.map(_.search(Some(query)))).map(_.flatten)
      } yield {
        available.toMap
      },
      defaultTimeout
    )
  }

  def search(query: String): Map[ArtifactId, Option[PluginMetadata]] = search(Option(query).filter(_.nonEmpty))

  def list(): Map[ArtifactId, Option[PluginMetadata]] = search(None)

  def listInstalled(): Seq[PluginId] =
    pluginManager.listInstalled()

  def installOrUpdate(plugin: Plugin): Unit = {
    logger.info(s"Installing plugin ${plugin.id.id}...")
    pluginManager.installOrUpdate(plugin)
    logger.info(s"Installation of plugin ${plugin.id.id} to the database is complete. " +
      s"System restart required for the plugin to be ready for usage.")
  }

  def install(pluginId: PluginId, bytes: Array[Byte]): InstallationResult = {
    logger.info(s"Installing plugin $pluginId...")
    validateInstall(pluginId, bytes) {
      pluginManager.install(Plugin(pluginId, None, bytes))
      val result = InstallationSuccess(pluginId.pluginName)
      logger.info(result.message)
      result
    }
  }

  def validateInstall(pluginId: PluginId, bytes: Array[Byte])(successCallback: => InstallationResult): InstallationResult = {
    val installedPlugins = listInstalled()
    val sameVersionAlreadyInstalled = installedPlugins.exists(_.equals(pluginId))
    val pluginName = pluginId.pluginName
    val differentVersionAlreadyInstalled = installedPlugins.exists(_.pluginName.equals(pluginName))

    if (sameVersionAlreadyInstalled) {
      val result = SameVersionInstalledAlready(pluginName)
      logger.info(result.message)
      result
    } else if (differentVersionAlreadyInstalled) {
      val result = DifferentVersionInstalledAlready(pluginName)
      logger.info(result.message)
      result
    } else {
      val valid = new PluginValidatorAtInstallation().validate(bytes, pluginId)
      if (valid.success) successCallback else valid
    }
  }

  def update(existingPluginName: String, newVersion: PluginId.LocalFile, bytes: Array[Byte]): InstallationResult = {
    logger.info(s"Updating plugin $existingPluginName to version $newVersion...")
    val matchingPlugins = listInstalled().filter(plugin => plugin.pluginId.pluginName.startsWith(existingPluginName) && plugin.pluginId.isInstanceOf[LocalFile])
    if (matchingPlugins.size > 1) {
      val result = MultipleInstalledPluginsFoundForUpdate(existingPluginName)
      logger.info(result.message)
      result
    } else if (matchingPlugins.isEmpty) {
      val result = ZeroInstalledPluginsFoundForUpdate(existingPluginName)
      logger.info(result.message)
      result
    } else {
      val alreadyInstalled = matchingPlugins.head.asInstanceOf[LocalFile]
      pluginManager.update(alreadyInstalled, Plugin(newVersion, None, bytes))
      val result = UpdateSuccess(alreadyInstalled.pluginName)
      logger.info(result.message)
      result
    }
  }

  def installOrUpdateFromRepository(id: PluginId.Artifact): Try[Unit] =
    repositories.get(id.repository).map { repo =>
      Try {
        val plugin = Await.result(repo.get(id), defaultTimeout)
        pluginManager.install(plugin)
      }.recoverWith {
        case err =>
          logger.warn(err.getMessage)
          Failure(err)
      }
    }.getOrElse {
      Failure(new NotFoundException(s"Unknown plugin repository '${id.repository}"))
    }

  def installFromRepository(id: PluginId.Artifact): InstallationResult =
    try {
      repositories.get(id.repository).map { repo =>
        val plugin = Await.result(repo.get(id), defaultTimeout)
        validateInstall(plugin.id, plugin.bytes) {
          pluginManager.install(plugin)
          val result = InstallationSuccess(id.pluginName)
          logger.info(result.message)
          result
        }
      }.getOrElse {
        val result = RepositoryNotFound(id.repository, id.pluginName)
        logger.info(result.message)
        result
      }
    } catch {
      case err: Exception =>
        logger.warn(err.getMessage)
        val result = PluginNotFound(id.pluginName)
        logger.info(result.message)
        result
    }

  def updateFromRepository(id: PluginId.Artifact): InstallationResult = {
    def doUpdate(repo: PluginsRepository, newVersion: PluginId.Artifact, existing: PluginId): InstallationResult = {
      val newPlugin = Await.result(repo.get(newVersion), defaultTimeout)
      pluginManager.update(existing, newPlugin)
      UpdateSuccess(newPlugin.id.pluginName)
    }

    repositories.get(id.repository).map { repo =>
      val installedPlugins = getPluginsToUpdate(id.pluginName)
      val availablePlugins = getUpdateMatchingPlugins(id)

      val updateResult = new UpdateStrategy().doUpdate(repo, installedPlugins, availablePlugins, id.pluginName, doUpdate)

      logger.info(updateResult.message)
      updateResult
    }.getOrElse {
      val result = RepositoryNotFound(id.repository, id.pluginName)
      logger.info(result.message)
      result
    }
  }

  def getPluginsToUpdate(pluginName: String): Seq[PluginId] = {
    val pluginNameCriteria = if (pluginName.endsWith("*")) pluginName.dropRight(1) else pluginName
    listInstalled().filter(plugin => plugin.pluginId.pluginName.startsWith(pluginNameCriteria) && plugin.pluginId.isInstanceOf[Artifact])
  }

  def getUpdateMatchingPlugins(id: PluginId.Artifact): Map[ArtifactId, Option[PluginMetadata]] = {
    if (id.pluginName.endsWith("*")) searchOfficial(id.pluginName.dropRight(1))
    else searchOfficial(id.pluginName)
      .filter(item => item._1.repository.get.equals(id.repository) && item._1.groupId.equals(id.groupId) && item._1.artifactId.equals(id.artifactId))
  }

  def uninstall(repositoryId: PluginSource.Value,
                groupId: String,
                artifactId: String,
                version: Option[String]): Try[Unit] = {
    logger.info(s"Uninstalling plugin $repositoryId:$groupId:$artifactId-$version")

    pluginManager.getPluginBy(repositoryId, groupId, artifactId, version) match {
      case Some(plugin) =>
        if (pluginManager.uninstall(plugin)) {
          pluginManager.revertPreviousVersionIfNecessary(plugin)
          Success()
        } else {
          Failure(PluginUninstallException(plugin, s"Only plugins in ${PluginStatus.READY_FOR_INSTALL} status can be uninstalled."))
        }
      case None => Failure(PluginNotFoundException(s"Plugin $repositoryId:$groupId:$artifactId-$version not found in the system."))
    }
  }

  def getLogo(repositoryId: String, artifactId: String): Option[File] =
    for {
      repo <- repositories.get(repositoryId)
      logo <- repo.getLogo(artifactId)
    } yield logo

  def attachMetadata(pluginId: PluginId): PluginDto = {
    val repoId = pluginId.pluginSource
    val metadata = for {
      repo <- repositories.get(repoId)
      pluginsMeta <- repo.getMetadata(pluginId.toArtifactId)
    } yield pluginsMeta
    PluginDto(pluginId, metadata)
  }

  def attachStatus: PluginDto => PluginDto = {
    val statuses: Map[String, PluginStatus.Value] = pluginManager.fetchPluginStatuses()
    pluginDto => pluginDto.copy(status = statuses.get(pluginDto.plugin.pluginName))
  }

  def extend(data: Seq[PluginId]): Seq[PluginDto] = data.map(attachMetadata).map(attachStatus)

  @PreDestroy
  def shutdown(): Unit = {
    logger.info("Shutting down LocalPluginManager...")
    Await.ready(
      Future.sequence(repositories.values.collect {
        case nexus: NexusPluginRepository => nexus.shutdown()
      }),
      repositories.size * defaultTimeout
    )
  }

}

object PluginService {
  // TODO: get this from configuration
  val defaultTimeout: Duration = 180 seconds

  def serversFromConfig(pluginsConfig: PluginManagerProperties): Map[String, Try[NexusServerConfig]] =
    Option(pluginsConfig.getServers) match {
      case Some(servers) => servers.asScala.collect(server =>
        server.getName -> NexusServerConfig.fromConfig(server)).toMap
      case None => Map.empty
    }

  def repositoriesFromConfig(pluginsConfig: PluginManagerProperties, servers: Map[String, Try[NexusServerConfig]])
                            (implicit productConfig: ProductConfig): Map[String, Try[NexusRepositoryConfig]] =
    Option(pluginsConfig.getRepositories) match {
      case Some(repositories) => repositories.asScala.map(repo =>
        repo.getName -> NexusRepositoryConfig.fromConfig(repo.getName)(repo, servers)).toMap
      case None => Map.empty
    }

  def configuredRepositories(pluginsConfig: PluginManagerProperties)
                            (implicit productConfig: ProductConfig): List[Try[NexusPluginRepository]] =
    repositoriesFromConfig(pluginsConfig, serversFromConfig(pluginsConfig))
      .map { case (name, tryConfig) =>
        tryConfig.map(config => NexusPluginRepository.memCached(name, config))
      }.toList
}

class UpdateStrategy {

  def areSameVersions(installed: PluginId, available: ArtifactId): Boolean = installed.toArtifactId.version.equals(available.version)

  def doUpdate(repo: PluginsRepository,
               installedPlugins: Seq[PluginId],
               availablePlugins: Map[ArtifactId, Option[PluginMetadata]],
               pluginName: String,
               doUpdate: (PluginsRepository, PluginId.Artifact, PluginId) => InstallationResult): InstallationResult = {
    (installedPlugins, availablePlugins) match {
      case (installed, _) if installed.size > 1 => MultipleInstalledPluginsFoundForUpdate(pluginName)
      case (installed, _) if installed.isEmpty => ZeroInstalledPluginsFoundForUpdate(pluginName)
      case (_, available) if available.size > 1 => MultipleAvailablePluginsFoundForUpdate(pluginName, repo.name)
      case (_, available) if available.isEmpty => ZeroAvailablePluginsFoundForUpdate(pluginName, repo.name)
      case (installed, available) if installed.size == 1 && available.size == 1 && areSameVersions(installed.head, available.head._1) => PluginAlreadyAtLatestVersion(pluginName)
      case (installed, available) if installed.size == 1 && available.size == 1 =>
        doUpdate(repo, availablePlugins.head._1.toArtifact, installed.head.pluginId)
    }
  }

}

@HttpResponseCodeResult(statusCode = 501)
case class PluginUninstallException(plugin: DbPlugin, message: String) extends DeployitException(s"Unable to uninstall plugin $plugin. $message")

@HttpResponseCodeResult(statusCode = 404)
case class PluginNotFoundException(message: String) extends DeployitException(message)

@HttpResponseCodeResult(statusCode = 400)
case class JARNotFoundException(message: String) extends DeployitException(message)

class InstallationResult(val success: Boolean, val message: String)

case class InstallationSuccess(pluginName: String)
  extends InstallationResult(success = true, message = s"Installation of $pluginName succeeded")

case class SameVersionInstalledAlready(pluginName: String)
  extends InstallationResult(success = false, message = s"Installation of $pluginName aborted. The same plugin version already exists in the system")

case class DifferentVersionInstalledAlready(pluginName: String)
  extends InstallationResult(success = false, message = s"Could not install $pluginName because another version of this plugin is already installed.")

case class ValidationFailed(pluginName: String)
  extends InstallationResult(success = false, message = s"Installation aborted. Types that $pluginName brings are not compatible with the current type system.")

case class RepositoryNotFound(repositoryName: String, pluginName: String)
  extends InstallationResult(success = false, message = s"Installation of $pluginName aborted. Unknown plugin repository: $repositoryName.")

case class PluginNotFound(pluginName: String)
  extends InstallationResult(success = false, message = s"Installation of $pluginName aborted. Unknown plugin in repository.")

case class UpdateSuccess(pluginName: String)
  extends InstallationResult(success = true, message = s"Update of plugin $pluginName succeeded")

case class UpdateFailed(pluginName: String)
  extends InstallationResult(success = false, message = s"The plugin you're trying to update ($pluginName) doesn't exist in the system.")

case class ZeroInstalledPluginsFoundForUpdate(pluginName: String)
  extends InstallationResult(success = false, message = s"Unable to find a plugin to update that would match your input criteria: ($pluginName)")

case class MultipleInstalledPluginsFoundForUpdate(pluginName: String)
  extends InstallationResult(success = false, message = s"Multiple installed plugins found for update for your input criteria: ($pluginName). Please narrow down the criteria.")

case class MultipleAvailablePluginsFoundForUpdate(pluginName: String, repoName: String)
  extends InstallationResult(success = false, message = s"Multiple available plugins found for update in remote repository $repoName for your input criteria: ($pluginName). Please narrow down the criteria.")

case class ZeroAvailablePluginsFoundForUpdate(pluginName: String, repoName: String)
  extends InstallationResult(success = false, message = s"Unable to find a plugin in remote repository $repoName to update the installed plugin with. Please update your input criteria ($pluginName)")

case class PluginAlreadyAtLatestVersion(pluginName: String)
  extends InstallationResult(success = true, message = s"Installed plugin already at highest version. Update will not be performed.")