package com.xebialabs.plugin.manager.repository

import com.xebialabs.plugin.manager.metadata.{ArtifactId, PluginMetadata}
import com.xebialabs.plugin.manager.repository.storage.PluginMetadataStorage
import com.xebialabs.plugin.manager.repository.storage.PluginMetadataStorageException.UnknownPlugin
import com.xebialabs.plugin.manager.{Plugin, PluginId}
import grizzled.slf4j.Logging
import org.joda.time.DateTime

import java.io.{ByteArrayInputStream, File}
import scala.concurrent.Future


abstract class CachedPluginRepository[R](toPluginId: R => Option[PluginId.Artifact]) extends PluginsRepository with Logging {
  protected var timestamp: Option[DateTime] = None

  override def lastUpdated: Option[DateTime] = timestamp

  def cache: PluginMetadataStorage

  def getFromRemote(id: PluginId.Artifact): Future[ByteArrayInputStream]

  def searchFromRemote(query: Option[String]): Future[Seq[R]]

  override def get(id: PluginId.Artifact): Future[Plugin] = cache.get(id) match {
    case None =>
      Future.failed(UnknownPlugin(name, id))

    case Some(PluginMetadata(metadata, _)) =>
      getFromRemote(id).map(content => Plugin(id, metadata, content.readAllBytes()))
  }

  override def getLogo(artifactId: String): Option[File] = cache.getLogo(artifactId)

  override def update(): Future[DateTime] = this.synchronized {
    logger.debug(s"Fetching metadata for plugin repository ${name}")
    for {
      metadata <- fetchMetadata
      pluginIds = metadata.keySet.toSeq
      mapped = pluginIds.map(_.toArtifact(name))
      _ <- updateLogos(pluginIds.map(_.artifactId))
    } yield {
      cache.clear()
      cache.store(mapped, metadata)
      timestamp = Some(DateTime.now)
      timestamp.get
    }
  }

  override def search(query: Option[String]): Future[Map[ArtifactId, Option[PluginMetadata]]] =
    Future.successful(cache.search(query))

  override def getMetadata(id: ArtifactId): Option[PluginMetadata] =
    cache.get(id)

  def updateMetadata(): Future[Seq[PluginId.Artifact]] = {
    logger.trace("updateMetadata()")
    for {
      metadata <- fetchMetadata
      mapped = metadata.keySet.toSeq.map(_.toArtifact(name))
    } yield {
      cache.clear()
      cache.store(mapped, metadata)
      mapped
    }
  }

  protected def updateLogos(artifactIds: Seq[String]): Future[Unit] = {
    logger.trace("updateLogos()")
    fetchLogos
      .map(cache.storeLogos(artifactIds)).map(_ => ())
  }

}
