package com.xebialabs.plugin.manager.validator

import com.xebialabs.plugin.manager.PluginId
import com.xebialabs.plugin.manager.config.ConfigWrapper
import com.xebialabs.xlplatform.synthetic.xml.SyntheticXmlDocument
import org.apache.commons.io.FileUtils

import java.io.File
import java.nio.charset.StandardCharsets
import java.util.jar.JarFile
import java.util.zip.{ZipEntry, ZipFile}
import scala.util.{Failure, Try}

abstract class SyntheticExtractor(val extension: String) {
  def extractFrom(file: File, pluginId: PluginId): Option[Try[SyntheticXmlDocument]]

  def readSynthetic(file: File): Option[Try[SyntheticXmlDocument]] = {
    Try(new JarFile(file)).toOption.flatMap(jarFile => {
      val syntheticXml = Option(jarFile.getEntry("synthetic.xml"))
        .map(entry => new String(jarFile.getInputStream(entry).readAllBytes(), StandardCharsets.UTF_8))
        .map(syntheticString => Try(SyntheticXmlDocument.read(syntheticString)))

      jarFile.close()
      syntheticXml
    })
  }
}

class JarSyntheticExtractor extends SyntheticExtractor(extension = ConfigWrapper.EXTENSION_JAR) {
  def extractFrom(file: File, pluginId: PluginId): Option[Try[SyntheticXmlDocument]] = {
    readSynthetic(file)
  }
}

class XldpSyntheticExtractor extends SyntheticExtractor(extension = ConfigWrapper.EXTENSION_XLDP) {

  def extractFrom(file: File, pluginId: PluginId): Option[Try[SyntheticXmlDocument]] = {
    val pluginVersion = if (pluginId.pluginVersion.isDefined) "-" + pluginId.pluginVersion.get.id else ""
    val jarName = pluginId.pluginName + pluginVersion + ".jar"

    val xldpFile = new ZipFile(file)
    val jarEntry = getJarEntryFrom(xldpFile, jarName)

    val result = if (jarEntry.isDefined) {
      val jarInputStream = xldpFile.getInputStream(jarEntry.get)
      val pluginFile = ValidatorUtils.createFileInWorkFolderFrom(jarInputStream.readAllBytes(), pluginId.pluginName)
      jarInputStream.close()

      val synthetic = readSynthetic(pluginFile)

      FileUtils.deleteDirectory(pluginFile.getParentFile)
      synthetic
    } else {
      Some(Failure(new IllegalArgumentException("JAR file doesn't exist : " + jarName)))
    }

    xldpFile.close()
    result
  }

  def getJarEntryFrom(xldpFile: ZipFile, jarName: String): Option[ZipEntry] = {
    // when creating xldp on mac the jar inside can only be extracted by using its name without "./". For official
    // plugins (ziped by gradle) it's the other way around. I have no idea why, so hence this workaround.
    val jarNameStartingWithDotSlash = "./" + jarName
    val jarEntry = Option(xldpFile.getEntry(jarName))

    if (jarEntry.isDefined) jarEntry else Option(xldpFile.getEntry(jarNameStartingWithDotSlash))
  }
}
