package com.xebialabs.plugin.manager.repository.nexus

import akka.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import com.xebialabs.plugin.manager.config.Repository
import com.xebialabs.plugin.manager.config.NexusMetadataArtifact
import com.xebialabs.plugin.manager.metadata.Version
import com.xebialabs.plugin.manager.repository.nexus.NexusRepositoryConfig.MetadataId
import com.xebialabs.plugin.manager.service.ProductConfig
import com.xebialabs.plugin.manager.{PluginId, requireProperFilename}
import spray.json._

import scala.language.implicitConversions
import scala.util.{Failure, Success, Try}


case class NexusRepositoryConfig(server: NexusServerConfig,
                                 productVersion: Version,
                                 repositoryId: String,
                                 metadataArtifactId: MetadataId,
                                 listMethod: NexusRepositoryConfig.ListMethod.ByGroupId) {
  lazy val metadataVersion: Version = productVersion.copy(extra = None)
}

object NexusRepositoryConfig {

  def byGroupId(server: NexusServerConfig,
                productVersion: Version,
                repositoryId: String,
                metadataId: MetadataId,
                groupId: String,
                packagingType: Option[String] = None,
                classifier: Option[String] = None): NexusRepositoryConfig = {
    NexusRepositoryConfig(server, productVersion, repositoryId, metadataId, ListMethod.ByGroupId(groupId, packagingType, classifier))
  }

  case class MetadataId(groupId: String, artifactId: String) {
    requireProperFilename(groupId, "groupId")
    requireProperFilename(artifactId, "artifactId")
  }

  object MetadataId {

    trait Protocol extends SprayJsonSupport with DefaultJsonProtocol {
      implicit val metadataIdWriter: JsonWriter[MetadataId] = metadataId =>
        JsObject(
          "group-id" -> metadataId.groupId.toJson,
          "artifact-id" -> metadataId.artifactId.toJson
        )
    }

    implicit class MetadataIdOps(val metadataId: MetadataId) extends AnyRef {
      def id: String = s"${metadataId.groupId}:${metadataId.artifactId}"

      def toArtifact(repoName: String, version: Version, classifier: Option[String] = None): PluginId.Artifact =
        PluginId.Artifact.parsed(repoName, metadataId.groupId, metadataId.artifactId, version, classifier)
    }

  }

  sealed trait ListMethod

  object ListMethod {

    case class ByGroupId(groupId: String, packaging: Option[String], classifier: Option[String]) extends ListMethod {
      requireProperFilename(groupId, "groupId")
    }

    object ByGroupId {
      def fromConfig(repositoryConfig: Repository): Try[ByGroupId] = {
       Try(Option(repositoryConfig.getNexusGroupId).get).map { groupId =>
          val packagingType = Option(repositoryConfig.getNexusPackagingType)
         ByGroupId(groupId, packagingType, None)
        }
      }
    }

  }

  def fromConfig(name: String)(repositoryConfig: Repository, servers: Map[String, Try[NexusServerConfig]])
                (implicit productConfig: ProductConfig): Try[NexusRepositoryConfig] = {

    def metadataIdFromConfig(config: NexusMetadataArtifact): Try[MetadataId] = for {
      groupId <- Try(Option(config.getGroupId).get)
      artifactId <- Try(Option(config.getArtifactId).get)
    } yield MetadataId(groupId, artifactId)

    for {
      _ <- Try(Option(repositoryConfig.getEnabled).get).flatMap {
        case "true" => Success(())
        case _ => Failure(new IllegalArgumentException(s"Repository configuration '$name' is not enabled."))
      }

      server <- if (repositoryConfig.getServerRef != null && repositoryConfig.getServerRef.nonEmpty) {
        val serverRefName = repositoryConfig.getServerRef
        servers.get(serverRefName) match {
          case None => Failure(new RuntimeException(s"servers.$serverRefName is missing."))
          case Some(serverConfig) => serverConfig
        }
      } else {
        Failure(new IllegalArgumentException(s"Repository configuration '$name' is missing a server-ref property."))
      }
      listMethod <- ListMethod.ByGroupId.fromConfig(repositoryConfig)
      repositoryId <- Try(Option(repositoryConfig.getNexusRepositoryId).get)
      metadataId <- Try(Option(repositoryConfig.getNexusMetadataArtifact).get).flatMap(metadataIdFromConfig)
    } yield {
      NexusRepositoryConfig(server, productConfig.version, repositoryId, metadataId, listMethod)
    }
  }

  trait Protocol extends SprayJsonSupport with DefaultJsonProtocol
    with NexusServerConfig.Protocol
    with MetadataId.Protocol {

    implicit val nexusConfigWriter: RootJsonWriter[NexusRepositoryConfig] = config =>
      JsObject(
        "server" -> config.server.toJson,
        "repository-id" -> config.repositoryId.toJson,
        "nexus-metadata-artifact" -> config.metadataArtifactId.toJson,
        "repository-type" -> "nexus-by-group-id".toJson,
        "nexus-group-id" -> config.listMethod.groupId.toJson,
        "nexus-packaging-type" -> config.listMethod.packaging.toJson
      )
  }
}

