package com.xebialabs.plugin.manager.model

import com.xebialabs.plugin.manager.config.ConfigWrapper.{GROUP_ID_XLD, GROUP_ID_XLR}
import com.xebialabs.plugin.manager.metadata.XLProduct
import com.xebialabs.plugin.manager.util.ChecksumUtil.calculateChecksum
import com.xebialabs.plugin.manager.rest.api.PluginSource.{LOCAL, XLD_OFFICIAL, XLR_OFFICIAL}
import com.xebialabs.plugin.manager.rest.api.PluginStatus.READY_FOR_INSTALL
import com.xebialabs.plugin.manager.util.VersionComparisonUtil

import java.nio.file.{Files, Path}
import scala.util.matching.Regex

case class FilePlugin(name: String,
                      version: Option[String],
                      extension: String,
                      filePath: Path) {}

object FilePlugin {
  lazy val pattern: Regex = ("""(.+)-(\d.*)\.([a-z]*)""").r
  lazy val patternNoVersion: Regex = ("""(.+)\.([a-z]*)""").r
  lazy val patternVersionWithNightly: Regex = ("""(.+)-(\d.*-\d.*)\.([a-z]*)""").r

  def apply(file: Path): FilePlugin = {
    file.getFileName.toString match {
      case patternVersionWithNightly(name, version, extension) =>
        new FilePlugin(name, Option(version), extension, file)
      case pattern(name, version, extension) =>
        new FilePlugin(name, Option(version), extension, file)
      case patternNoVersion(name, extension) =>
        new FilePlugin(name, Option.empty, extension, file)
      case _ =>
        new FilePlugin(null, Option.empty, null, null)
    }
  }

  implicit class FilePluginOps(val fsPlugin: FilePlugin) extends AnyVal {
    def isLowerVersionThan(dbPlugin: DbPlugin): Boolean = VersionComparisonUtil.isLowerVersion(fsPlugin.version.get, dbPlugin.version.getOrElse("0"))

    def isSameVersionLike(dbPlugin: DbPlugin): Boolean = VersionComparisonUtil.isSameVersion(fsPlugin.version.getOrElse("0"), dbPlugin.version.getOrElse("0"))

    def doesntExistIn(dbPlugins: Seq[DbPlugin]): Boolean = dbPlugins.forall(dbPlugin => !fsPlugin.name.equals(dbPlugin.name))

    def isHigherVersionThanAMatchIn(dbPlugins: Seq[DbPlugin]): Boolean = dbPlugins.exists(dbPlugin => fsPlugin.name.equals(dbPlugin.name) && dbPlugin.isLowerVersionThan(fsPlugin))

    def hasDifferentContentAndVersionThanAMatchIn(dbPlugins: Seq[DbPlugin]): Boolean = dbPlugins.exists(dbPlugin => fsPlugin.name.equals(dbPlugin.name) && (isContentDifferent(fsPlugin.filePath, dbPlugin.checksum) || !fsPlugin.isSameVersionLike(dbPlugin)))

    def isContentDifferent(filePluginPath: Path, dbPluginChecksum: String): Boolean = {
      val filePluginChecksum = calculateChecksum(Files.readAllBytes(filePluginPath))
      !filePluginChecksum.equals(dbPluginChecksum)
    }

    def toOfficialDbPlugin(product: XLProduct): DbPlugin = {
      val groupId = if (product == XLProduct.XLDeploy) GROUP_ID_XLD else GROUP_ID_XLR
      val source = if (product == XLProduct.XLDeploy) XLD_OFFICIAL else XLR_OFFICIAL
      val bytes = Files.readAllBytes(fsPlugin.filePath)
      new DbPlugin(0, fsPlugin.name, fsPlugin.version, fsPlugin.extension, groupId, READY_FOR_INSTALL, source, calculateChecksum(bytes), Some(0), Some(bytes))
    }

    def toLocalDbPlugin: DbPlugin = {
      val bytes = Files.readAllBytes(fsPlugin.filePath)
      new DbPlugin(0, fsPlugin.name, fsPlugin.version, fsPlugin.extension, LOCAL.toString, READY_FOR_INSTALL, LOCAL, calculateChecksum(bytes), Some(0), Some(bytes))
    }
  }
}
