package com.xebialabs.plugin.manager.validator

import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, Type}
import com.xebialabs.plugin.manager.PluginId
import com.xebialabs.plugin.manager.config.ConfigWrapper
import grizzled.slf4j.Logging
import org.apache.commons.io.FileUtils

import java.io.File
import java.util.zip.ZipFile
import scala.util.{Failure, Success, Try}

abstract class PluginValidator(syntheticValidator: SyntheticValidator, syntheticExtractor: SyntheticExtractor) extends Logging {

  def validate(bytes: Array[Byte], pluginId: PluginId): ValidationResult = {
    val pluginFile = ValidatorUtils.createFileInWorkFolderFrom(bytes, "validation-" + pluginId.pluginName)

    val zipFile = Try(new ZipFile(pluginFile))
    val result = if (zipFile.isSuccess) {
      zipFile.get.close()
      validateInternal(pluginFile, pluginId)
    } else InvalidArchive(pluginId.pluginName)

    FileUtils.deleteDirectory(pluginFile.getParentFile)
    result
  }

  private def validateInternal(pluginFile: File, pluginId: PluginId): ValidationResult = {
    val syntheticOpt = syntheticExtractor.extractFrom(pluginFile, pluginId)
    syntheticOpt match {
      case Some(Success(synthetic)) => syntheticValidator.validate(synthetic)
      case Some(Failure(exception)) =>
        logger.debug("Synthetic.xml validation failed. Invalid xml. Message: " + exception.getMessage)
        SyntheticXmlNotValid(pluginId.pluginName)
      case None => PluginValid()
    }
  }
}

trait DescriptorRegistryWrapper {
  def exists(wanted: Type): Boolean
}

class DefaultDescriptorRegistryWrapper extends DescriptorRegistryWrapper {
  def exists(wanted: Type): Boolean = DescriptorRegistry.exists(wanted)
}

class PluginValidatorAtInstallation(syntheticValidator: SyntheticValidator, syntheticExtractor: SyntheticExtractor) extends PluginValidator(syntheticValidator = syntheticValidator, syntheticExtractor = syntheticExtractor) {
  def this(syntheticExtractor: SyntheticExtractor, descriptorRegistryWrapper: DescriptorRegistryWrapper) = {
    this(new SyntheticValidatorAtInstallation(descriptorRegistryWrapper), syntheticExtractor)
  }

  def this() = {
    this(new SyntheticValidatorAtInstallation(new DefaultDescriptorRegistryWrapper),
      if (ConfigWrapper.EXTENSION_JAR.equals(ConfigWrapper.extension)) new JarSyntheticExtractor else new XldpSyntheticExtractor)
  }
}

class PluginValidatorAtUpdate extends PluginValidator(syntheticValidator = new SyntheticValidatorAtUpdate, syntheticExtractor = new JarSyntheticExtractor)
