package com.xebialabs.plugin.manager.repository.sql

import com.xebialabs.plugin.manager.model.DbPlugin
import com.xebialabs.plugin.manager.rest.api.{PluginSource, PluginStatus}
import com.xebialabs.plugin.manager.repository.sql.MigrationsSchema.{XLPM_PLUGIN, XLPM_PLUGIN_BYTES}
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.namedparam.{MapSqlParameterSource, NamedParameterJdbcTemplate}
import org.springframework.jdbc.core.support.SqlLobValue
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.jdbc.support.GeneratedKeyHolder
import org.springframework.util.StreamUtils.copyToByteArray

import java.sql.{Connection, ResultSet, Types}
import scala.jdk.CollectionConverters.{ListHasAsScala, MapHasAsJava}
import scala.util.{Failure, Success, Try, Using}

class SqlPluginRepository(jdbcTemplate: JdbcTemplate) extends PluginQueries with Logging {

  implicit lazy val namedTemplate: NamedParameterJdbcTemplate = new NamedParameterJdbcTemplate(jdbcTemplate)
  lazy val dialect: Dialect = Dialect(jdbcTemplate)

  def insert(plugin: DbPlugin): Unit = {
    val keyHolder = new GeneratedKeyHolder
    val mapSqlParameterSource = new MapSqlParameterSource()
    mapSqlParameterSource.addValue(XLPM_PLUGIN_BYTES.bytes, new SqlLobValue(plugin.bytes.get), Types.BLOB)
    val keyColumnNames = Array[String] {
      dialect.withDialect("id")
    } // leave lowercase "id" for compatibility with Postgres
    namedTemplate.update(INSERT_PLUGIN_BYTES, mapSqlParameterSource, keyHolder, keyColumnNames)

    namedTemplate.update(INSERT_PLUGIN, new MapSqlParameterSource(params(
      XLPM_PLUGIN.name -> plugin.name,
      XLPM_PLUGIN.version -> plugin.version.orNull,
      XLPM_PLUGIN.extension -> plugin.extension,
      XLPM_PLUGIN.groupId -> plugin.groupId,
      XLPM_PLUGIN.installationStatus -> plugin.installationStatus.toString,
      XLPM_PLUGIN.source -> plugin.source.toString,
      XLPM_PLUGIN.checksum -> plugin.checksum,
      XLPM_PLUGIN.pluginBytesId -> keyHolder.getKey.intValue).asJava))
  }

  def update(pluginFromApi: DbPlugin, pluginToUpdate: DbPlugin): Unit = {
    val mapSqlParameterSource = new MapSqlParameterSource()
    mapSqlParameterSource.addValue(XLPM_PLUGIN_BYTES.bytes, new SqlLobValue(pluginFromApi.bytes.get), Types.BLOB)
    mapSqlParameterSource.addValue(XLPM_PLUGIN_BYTES.id, pluginToUpdate.pluginBytesId.get, Types.BIGINT)
    namedTemplate.update(UPDATE_PLUGIN_BYTES, mapSqlParameterSource)

    namedTemplate.update(UPDATE_PLUGIN, new MapSqlParameterSource(params(
      XLPM_PLUGIN.version -> pluginFromApi.version.orNull,
      XLPM_PLUGIN.groupId -> pluginFromApi.groupId,
      XLPM_PLUGIN.installationStatus -> pluginFromApi.installationStatus.toString,
      XLPM_PLUGIN.source -> pluginFromApi.source.toString,
      XLPM_PLUGIN.checksum -> pluginFromApi.checksum,
      XLPM_PLUGIN.id -> pluginToUpdate.id).asJava))
  }

  def delete(plugin: DbPlugin): Int = {
    namedTemplate.update(DELETE_PLUGIN, new MapSqlParameterSource(params(
      XLPM_PLUGIN.id -> plugin.id).asJava))
    namedTemplate.update(DELETE_PLUGIN_BYTES, new MapSqlParameterSource(params(
      XLPM_PLUGIN_BYTES.id -> plugin.pluginBytesId.get).asJava))
  }

  def getByName(pluginName: String): Seq[DbPlugin] = {
    Try(
      namedTemplate.query(SELECT_BY_NAME, params(XLPM_PLUGIN.name -> pluginName).asJava, dbPluginMapper).asScala.toSeq
    ).getOrElse(Seq.empty[DbPlugin])
  }

  def pluginTablesExist: Boolean = {
    Using(namedTemplate.getJdbcTemplate.getDataSource.getConnection()) { connection =>
      doesTableExist(connection, XLPM_PLUGIN.TABLE) || doesTableExist(connection, XLPM_PLUGIN.TABLE.toLowerCase())
    }.get
  }

  def doesTableExist(connection: Connection, tableName: String): Boolean = {
    val tables = connection.getMetaData.getTables(null, null, tableName, null)
    tables.next()
  }

  def getAllWithBytes: Seq[DbPlugin] = {
    val plugins = namedTemplate.query(SELECT_WITH_BYTES, dbPluginMapper)
    debug(s"Found the following plugins in the database: $plugins")
    plugins.asScala.toSeq
  }

  def getAllWithoutBytes: Seq[DbPlugin] = {
    val plugins = namedTemplate.query(SELECT, dbPluginWithoutBytesMapper)
    debug(s"Found the following plugins in the database: $plugins")
    plugins.asScala.toSeq
  }

  def updateAllPluginsStatusTo(status: PluginStatus.Value): Int = {
    namedTemplate.update(UPDATE_ALL_PLUGINS_STATUS, new MapSqlParameterSource("STATUS", status.toString))
  }

  def getPluginBy(repositoryId: PluginSource.Value, groupId: String, artifactId: String, version: Option[String]): Option[DbPlugin] = {
    val parameters = new MapSqlParameterSource(params(
      XLPM_PLUGIN.source -> repositoryId.toString,
      XLPM_PLUGIN.groupId -> groupId,
      XLPM_PLUGIN.name -> artifactId).asJava)
    if (version.isDefined) parameters.addValue(XLPM_PLUGIN.version, version.get)

    val query = if (version.isDefined) {
      SELECT_BY_REPO_GROUPID_ARTIFACTID_VERSION_STATUS
    } else {
      SELECT_BY_REPO_GROUPID_ARTIFACTID_STATUS
    }

    Try(namedTemplate.queryForObject(query, parameters, dbPluginWithoutBytesMapper)) match {
      case Failure(_) => None
      case Success(plugin) => Some(plugin)
    }
  }

  private def dbPluginMapper: RowMapper[DbPlugin] = (rs: ResultSet, _: Int) => {
    val bytesInputStream = rs.getBinaryStream(XLPM_PLUGIN_BYTES.bytes)
    Try(copyToByteArray(bytesInputStream)) match {
      case Success(bytes) => DbPlugin(
        rs.getLong(XLPM_PLUGIN.id),
        rs.getString(XLPM_PLUGIN.name),
        Option(rs.getString(XLPM_PLUGIN.version)),
        rs.getString(XLPM_PLUGIN.extension),
        rs.getString(XLPM_PLUGIN.groupId),
        PluginStatus.withName(rs.getString(XLPM_PLUGIN.installationStatus)),
        PluginSource.withName(rs.getString(XLPM_PLUGIN.source)),
        rs.getString(XLPM_PLUGIN.checksum),
        Some(rs.getLong(XLPM_PLUGIN.pluginBytesId)),
        Some(bytes))
      case Failure(ex) => throw new IllegalStateException("Failed to read plugin from the db", ex)
    }
  }

  private def dbPluginWithoutBytesMapper: RowMapper[DbPlugin] = (rs: ResultSet, _: Int) =>
    DbPlugin(
      rs.getLong(XLPM_PLUGIN.id),
      rs.getString(XLPM_PLUGIN.name),
      Option(rs.getString(XLPM_PLUGIN.version)),
      rs.getString(XLPM_PLUGIN.extension),
      rs.getString(XLPM_PLUGIN.groupId),
      PluginStatus.withName(rs.getString(XLPM_PLUGIN.installationStatus)),
      PluginSource.withName(rs.getString(XLPM_PLUGIN.source)),
      rs.getString(XLPM_PLUGIN.checksum),
      Some(rs.getLong(XLPM_PLUGIN.pluginBytesId)),
      None)

  def params(map: (String, _ <: Any)*): Map[String, Any] = map.toMap[String, Any]
}

object MigrationsSchema {

  abstract class Table(val TABLE: String)

  object XLPM_PLUGIN extends Table("XLPM_PLUGIN") {
    val id: String = "ID"
    val name: String = "NAME"
    val version: String = "VERSION"
    val extension: String = "EXTENSION"
    val groupId: String = "GROUP_ID"
    val installationStatus: String = "INSTALLATION_STATUS"
    val source: String = "SOURCE"
    val checksum: String = "CHECKSUM"
    val pluginBytesId: String = "PLUGIN_BYTES_ID"
  }

  object XLPM_PLUGIN_BYTES extends Table("XLPM_PLUGIN_BYTES") {
    val id: String = "ID"
    val bytes: String = "BYTES"
  }

}

trait PluginQueries {

  val SELECT: String =
    s"""SELECT ${XLPM_PLUGIN.id}, ${XLPM_PLUGIN.name}, ${XLPM_PLUGIN.version}, ${XLPM_PLUGIN.extension},
       | ${XLPM_PLUGIN.groupId}, ${XLPM_PLUGIN.installationStatus}, ${XLPM_PLUGIN.source}, ${XLPM_PLUGIN.checksum},
       | ${XLPM_PLUGIN.pluginBytesId}
       | FROM ${XLPM_PLUGIN.TABLE}""".stripMargin

  val SELECT_WITH_BYTES: String =
    s"""SELECT ${XLPM_PLUGIN.TABLE}.${XLPM_PLUGIN.id}, ${XLPM_PLUGIN.name}, ${XLPM_PLUGIN.version},
        ${XLPM_PLUGIN.extension}, ${XLPM_PLUGIN.groupId}, ${XLPM_PLUGIN.installationStatus}, ${XLPM_PLUGIN.source},
        ${XLPM_PLUGIN.checksum} , ${XLPM_PLUGIN.pluginBytesId}, ${XLPM_PLUGIN_BYTES.bytes}
        FROM ${XLPM_PLUGIN.TABLE}, ${XLPM_PLUGIN_BYTES.TABLE}
        WHERE ${XLPM_PLUGIN.TABLE}.${XLPM_PLUGIN.pluginBytesId} = ${XLPM_PLUGIN_BYTES.TABLE}.${XLPM_PLUGIN_BYTES.id}"""

  val SELECT_BY_NAME: String = SELECT_WITH_BYTES + s""" AND ${XLPM_PLUGIN.name} = :NAME"""

  val INSERT_PLUGIN_BYTES: String =
    s"""INSERT INTO ${XLPM_PLUGIN_BYTES.TABLE} (${XLPM_PLUGIN_BYTES.bytes}) VALUES (:BYTES)"""

  val INSERT_PLUGIN: String =
    s"""INSERT INTO ${XLPM_PLUGIN.TABLE} (${XLPM_PLUGIN.name}, ${XLPM_PLUGIN.version}, ${XLPM_PLUGIN.extension},
        ${XLPM_PLUGIN.groupId}, ${XLPM_PLUGIN.installationStatus}, ${XLPM_PLUGIN.source}, ${XLPM_PLUGIN.checksum},
        ${XLPM_PLUGIN.pluginBytesId}) VALUES (:NAME, :VERSION, :EXTENSION, :GROUP_ID, :INSTALLATION_STATUS, :SOURCE,
        :CHECKSUM, :PLUGIN_BYTES_ID)"""

  val UPDATE_PLUGIN_BYTES: String =
    s"""UPDATE ${XLPM_PLUGIN_BYTES.TABLE} SET ${XLPM_PLUGIN_BYTES.bytes} = :BYTES WHERE ${XLPM_PLUGIN_BYTES.id} = :ID"""

  val UPDATE_PLUGIN: String =
    s"""UPDATE ${XLPM_PLUGIN.TABLE} SET ${XLPM_PLUGIN.version} = :VERSION, ${XLPM_PLUGIN.groupId} = :GROUP_ID,
       ${XLPM_PLUGIN.installationStatus} = :INSTALLATION_STATUS, ${XLPM_PLUGIN.source} = :SOURCE,
       ${XLPM_PLUGIN.checksum} = :CHECKSUM
       WHERE ${XLPM_PLUGIN.id} = :ID"""

  val UPDATE_ALL_PLUGINS_STATUS =
    s"""UPDATE ${XLPM_PLUGIN.TABLE} SET ${XLPM_PLUGIN.installationStatus} = :STATUS"""

  val DELETE_PLUGIN_BYTES: String =
    s"""DELETE FROM ${XLPM_PLUGIN_BYTES.TABLE} WHERE ${XLPM_PLUGIN_BYTES.id} = :ID"""

  val DELETE_PLUGIN: String =
    s"""DELETE FROM ${XLPM_PLUGIN.TABLE} WHERE ${XLPM_PLUGIN.id} = :ID"""

  val SELECT_BY_REPO_GROUPID_ARTIFACTID_STATUS: String = SELECT +
    s""" WHERE ${XLPM_PLUGIN.source} = :SOURCE AND ${XLPM_PLUGIN.groupId} = :GROUP_ID AND ${XLPM_PLUGIN.name} = :NAME"""

  val SELECT_BY_REPO_GROUPID_ARTIFACTID_VERSION_STATUS: String =
    SELECT_BY_REPO_GROUPID_ARTIFACTID_STATUS + s""" AND ${XLPM_PLUGIN.version} = :VERSION"""

}
