package com.xebialabs.plugin.manager.startup

import com.xebialabs.deployit.booter.local.PluginVersions
import com.xebialabs.plugin.manager.repository.sql.VersionRepository
import org.springframework.jdbc.core.JdbcTemplate

/**
  * This class will give the embedding system an answer if it should proceed synching plugins or not.
  *
  * From plugin manager perspective there are 3 types of system start.
  *
  * 1. First start of a brand new installation (blank DB)
  * 2. First start of a new product version created with files from a previous installation (a.k.a. upgraded install)
  * 3. Regular start - not a first and not a second case. Same as a restart.
  *
  * Only in the last type of start is it safe (and very desirable) to do plugin synchronization.
  *
  * @param dataSource
  * @param componentName
  */
class SynchronizationCondition(jdbcTemplate: JdbcTemplate, componentName: String) {

  private val versionRepository = new VersionRepository(jdbcTemplate)

  PluginVersions.init()

  def shouldSync(): Boolean = if (versionRepository.xlVersionTableExists) isNotNewInstallationNorUpgrade else false

  private def isNotNewInstallationNorUpgrade = {
    val componentVersion = getComponentVersion
    val distributionVersion = getDistributionVersion

    val newInstallation = componentVersion == null
    val firstStartAfterUpgrade = componentVersion != null && !componentVersion.equals(distributionVersion)

    !newInstallation && !firstStartAfterUpgrade
  }

  private def getDistributionVersion = {
    val version = PluginVersions.getVersionFor(componentName)
    if (version.contains("-")) {
      version.substring(0, version.indexOf("-"))
    } else {
      version
    }
  }

  private def getComponentVersion = {
    versionRepository.selectComponentVersion(componentName) match {
      case Some(version) =>
        val versionString = version.getVersion
        if (versionString.contains("#")) {
          versionString.substring(0, versionString.indexOf("#"))
        } else {
          versionString
        }
      case None => null
    }
  }

}
