package com.xebialabs.plugin.manager.rest.json

import com.xebialabs.plugin.manager.metadata.{MetadataEntry, Version}
import com.xebialabs.plugin.manager.rest.dto.{PluginDto, PluginDtoList}
import org.springframework.stereotype.Component
import spray.json._

import java.io.{IOException, OutputStream, OutputStreamWriter}
import java.lang.annotation.Annotation
import java.lang.reflect.Type
import javax.ws.rs.core.{MediaType, MultivaluedMap}
import javax.ws.rs.ext.{MessageBodyWriter, Provider}
import javax.ws.rs.{Produces, WebApplicationException}

@Component
@Provider
@Produces(Array(MediaType.APPLICATION_JSON))
class PluginDtoJsonWriter extends MessageBodyWriter[PluginDtoList]
  with BasicFormats
  with Version.Protocol
  with StandardFormats
  with MetadataEntry.Protocol {

  override def isWriteable(`type`: Class[_], genericType: Type, annotations: Array[Annotation],
                           mediaType: MediaType): Boolean = classOf[PluginDtoList].isAssignableFrom(`type`)

  override def getSize(viewList: PluginDtoList, `type`: Class[_], genericType: Type, annotations: Array[Annotation],
                       mediaType: MediaType): Long = -1

  @throws[IOException]
  @throws[WebApplicationException]
  override def writeTo(response: PluginDtoList, `type`: Class[_], genericType: Type, annotations: Array[Annotation],
                       mediaType: MediaType, httpHeaders: MultivaluedMap[String, AnyRef],
                       entityStream: OutputStream): Unit = {

    val writer = new OutputStreamWriter(entityStream)
    writer.write(response.plugins.map(pluginsResult).toJson.compactPrint)
    writer.flush()
  }

  def pluginsResult(result: PluginDto): JsObject = {
    val js = JsObject(
      "plugin" -> JsObject(
        "groupId" -> result.plugin.groupId.toJson,
        "artifactId" -> result.plugin.pluginName.toJson
      ),
      "versions" -> JsArray(JsObject(
        "version" -> result.plugin.pluginVersion.toJson,
        "repository" -> result.plugin.pluginSource.toJson
      )),
      "metadata" -> result.metadata.map(_.metadata).toJson,
      "hasLogo" -> result.metadata.exists(_.hasLogo).toJson
    )

    if (result.status.isDefined) {
      JsObject(js.fields + ("status" -> result.status.get.toString.toJson))
    } else js
  }

}
