package com.xebialabs.plugin.manager.service

import com.typesafe.config.{Config, ConfigFactory}
import com.xebialabs.deployit.server.api.upgrade.{Version => PlatformVersion}
import com.xebialabs.plugin.manager._
import com.xebialabs.plugin.manager.metadata.{Version, XLProduct}
import com.xebialabs.plugin.manager.repository.PluginsRepository
import com.xebialabs.plugin.manager.repository.nexus.NexusRepositoryConfig.ListMethod
import com.xebialabs.plugin.manager.repository.nexus.{NexusPluginRepository, NexusRepositoryConfig}
import grizzled.slf4j.Logging

import scala.collection.mutable
import scala.language.postfixOps
import scala.util.{Failure, Success}


class LocalPluginService(config: Config, product: XLProduct, version: Version) extends PluginService with Logging {

  def this(config: Config, product: String, version: PlatformVersion) {
    this(
      config,
      XLProduct.fromString(product),
      Version.fromPlatform(version)
    )
  }

  def this(config: Config, product: String, version: String) {
    this(
      config,
      XLProduct.fromString(product),
      Version.fromString(version).getOrElse(throw new IllegalArgumentException(s"Wrong version format: $version"))
    )
  }

  protected implicit val productConfig: ProductConfig = ProductConfig(product, version.copy(extra = None))

  override val repositories: mutable.Map[String, PluginsRepository] =
    mutable.Map.empty ++ {
      PluginService.configuredRepositories(config).flatMap {
        case Success(repo) => List(repo.name -> repo)
        case Failure(ex) =>
          logger.warn(ex)
          List.empty
      }.toMap
    }

  override def addRepository(repository: PluginsRepository): Boolean = {
    repositories.get(repository.name).map(_ => false).getOrElse {
      logger.info(s"Adding new repository; '${repository.name}' -> $repository")
      this.repositories += repository.name -> repository
      this.pluginManagerInstance = updatePluginManager()
      true
    }
  }

  override def deleteRepository(name: String): Boolean = {
    repositories.get(name).exists { _ =>
      logger.info(s"Removing repository '$name'")
      repositories -= name
      this.pluginManagerInstance = updatePluginManager()
      true
    }
  }


  override def pluginManager: PluginManager = pluginManagerInstance

  private var pluginManagerInstance: PluginManager = updatePluginManager()

  private def updatePluginManager(): LocalPluginManager = new LocalPluginManager(repositories.collect {
    case (name, NexusPluginRepository(_, NexusRepositoryConfig(_, _, _, _, ListMethod.ByGroupId(groupId, _, _)), _)) =>
      name -> groupId
  }.toMap)

}

object LocalPluginService {
  def empty(implicit productConfig: ProductConfig) = new LocalPluginService(ConfigFactory.empty(), productConfig.product, productConfig.version)
}