package com.xebialabs.plugin.manager.repository.nexus

import akka.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import com.typesafe.config.{Config, ConfigException}
import com.xebialabs.plugin.manager.metadata.{ArtifactId, Version}
import com.xebialabs.plugin.manager.repository.nexus.NexusRepositoryConfig.ListMethod.{ByArtifactPrefix, ByGroupId}
import com.xebialabs.plugin.manager.{PluginId, requireNonEmptyNoColumnsAndNoSpaces}
import spray.json._

import scala.language.implicitConversions
import scala.util.{Failure, Try}


case class NexusRepositoryConfig(server: NexusServerConfig,
                                 repositoryId: String,
                                 metadataArtifactId: ArtifactId,
                                 xlComponent: String,
                                 listMethod: NexusRepositoryConfig.ListMethod)

object NexusRepositoryConfig {
  implicit class NexusRepositoryConfigOps(val config: NexusRepositoryConfig) extends AnyVal {
    def metadataId(forRepo: String): PluginId.Artifact =
      config.metadataArtifactId.toArtifact(forRepo, Version.Constant(2018, Some(3), Some(6), Some("-1442")))
  }

  def byGroupId(server: NexusServerConfig,
                repositoryId: String,
                metadataId: ArtifactId,
                xlComponent: String,
                groupId: String,
                packagingType: Option[String] = None,
                classifier: Option[String] = None): NexusRepositoryConfig =
    NexusRepositoryConfig(server, repositoryId, metadataId, xlComponent, ListMethod.ByGroupId(groupId, packagingType, classifier))

  // Note: currently unsupported.
  def byPrefix(server: NexusServerConfig,
               repositoryId: String,
               metadataId: ArtifactId,
               xlComponent: String,
               prefix: String): NexusRepositoryConfig =
    NexusRepositoryConfig(server, repositoryId, metadataId, xlComponent, ListMethod.ByArtifactPrefix(prefix))

  sealed trait ListMethod

  object ListMethod {

    case class ByGroupId(groupId: String, packaging: Option[String], classifier: Option[String]) extends ListMethod {
      requireNonEmptyNoColumnsAndNoSpaces(groupId, "groupId")
    }

    // Note: currently unsupported
    case class ByArtifactPrefix(prefix: String) extends ListMethod {
      requireNonEmptyNoColumnsAndNoSpaces(prefix, "prefix")
    }

    object ByGroupId {
      def fromConfig(repositoryConfig: Config): Try[ByGroupId] = {
       Try(repositoryConfig.getString("nexus-group-id")).map { groupId =>
          val packagingType = Try(repositoryConfig.getString("nexus-packaging-type")).toOption
         ByGroupId(groupId, packagingType, None)
        }
      }
    }

    object ByArtifactPrefix {
      def fromConfig(repositoryConfig: Config): Try[ByArtifactPrefix] =
        Try(repositoryConfig.getString("nexus-artifact-prefix"))
          .map(ByArtifactPrefix.apply)
    }

    def fromConfig(repositoryConfig: Config): Try[ListMethod] = {
      Try(repositoryConfig.getString("repository-type")).flatMap {
        case "nexus-by-group-id" => ByGroupId.fromConfig(repositoryConfig)
        case "nexus-by-prefix" => ByArtifactPrefix.fromConfig(repositoryConfig)
        case unknownRepoType =>
          Failure(new ConfigException.WrongType(repositoryConfig.origin(), "repository-type", "'nexus-by-group-id' or 'nexus-by-prefix'", unknownRepoType))
      }
    }
  }


  def fromConfig(name: String)(repositoryConfig: Config, servers: Map[String, Try[NexusServerConfig]]): Try[NexusRepositoryConfig] = {
    for {
      _ <- Try(repositoryConfig.getBoolean("enabled")).filter(identity)
      server <- if (repositoryConfig.hasPath("server-ref")) {
        val serverRefName = repositoryConfig.getString("server-ref")
        servers.get(serverRefName) match {
          case None => Failure(new ConfigException.Missing(s"servers.$serverRefName"))
          case Some(serverConfig) => serverConfig
        }
      } else {
        NexusServerConfig.fromConfig(repositoryConfig.getConfig("server"))
      }
      listMethod <- ListMethod.fromConfig(repositoryConfig)
      repositoryId <- Try(repositoryConfig.getString("nexus-repository-id"))
      xlComponent <- Try(repositoryConfig.getString("xl-component"))
      metadataArtifactId <- Try(repositoryConfig.getConfig("nexus-metadata-artifact")).flatMap(ArtifactId.fromConfig(Some(name)))
    } yield {
      NexusRepositoryConfig(server, repositoryId, metadataArtifactId.copy(repository = Some(name)), xlComponent, listMethod)
    }
  }

  trait Protocol extends SprayJsonSupport with DefaultJsonProtocol
    with NexusServerConfig.Protocol
    with ArtifactId.Protocol {

    implicit val nexusConfigWriter: RootJsonWriter[NexusRepositoryConfig] = config => {
      val base = Map(
        "server" -> config.server.toJson,
        "repository-id" -> config.repositoryId.toJson,
        "nexus-metadata-artifact" -> config.metadataArtifactId.toJson,
        "xl-component" -> config.xlComponent.toJson,
        "repository-type" -> {
          config.listMethod match {
            case _: ByGroupId => "nexus-by-group-id".toJson
            case _: ByArtifactPrefix => "nexus-by-prefix".toJson
          }
        }
      )
      val extra = config.listMethod match {
        case ByGroupId(groupId, packagingType, classifier) =>
          Map(
            "nexus-group-id" -> groupId.toJson,
            "nexus-packaging-type" -> packagingType.toJson
          )
        case ByArtifactPrefix(prefix) =>
          Map(
            "nexus-artifact-prefix" -> prefix.toJson
          )
      }
      JsObject(base ++ extra)
    }
  }
}

