package com.xebialabs.plugin.manager.rest.api

import akka.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import com.xebialabs.plugin.manager.PluginId
import com.xebialabs.plugin.manager.metadata.{ArtifactId, ExtendedMetadata, MetadataEntry, PluginsMetadata}
import com.xebialabs.plugin.manager.repository.PluginsRepository
import com.xebialabs.plugin.manager.repository.nexus.{NexusPluginRepository, NexusRepositoryConfig}
import spray.json._

import scala.util.{Failure, Success, Try}

case class InstallFromRepositoryOutcome(id: PluginId, outcome: Try[Option[ExtendedMetadata]])

trait PluginManagerProtocol extends SprayJsonSupport with DefaultJsonProtocol
  with MetadataEntry.Protocol
  with NexusRepositoryConfig.Protocol
  with PluginsMetadata.Protocol
  with PluginId.Protocol {

  def pluginsResult(result: (ArtifactId, PluginsMetadata)): JsObject = result match {
    case (id, plugins) =>
      JsObject(
        "plugin" -> id.toJson,
        "versions" -> plugins.versions.map { v =>
          JsObject(
            "version" -> v.map(_.id).toJson,
            "repository" -> id.repository.toJson
          )
        }.toJson,
        "metadata" -> plugins.metadata.toJson,
        "hasLogo" -> plugins.hasLogo.toJson
      )
  }

  implicit val pluginRepositoryWriter: JsonWriter[PluginsRepository] = jsonWriter[PluginsRepository] { (repo: PluginsRepository) =>
    JsObject(
      "repositoryId" -> repo.name.toJson,
      "lastUpdated" -> repo.lastUpdated.fold[JsValue](JsNull)(_.getMillis.toJson),
      "repositoryType" -> {
        repo match {
          case _: NexusPluginRepository => "nexus".toJson
          case _ => "unknown".toJson
        }
      },
      "config" -> {
        repo match {
          case nexus: NexusPluginRepository => nexus.config.toJson // TODO
          case _ => JsNull
        }
      }
    )
  }


  implicit def tryJsonWriter[T](implicit w: JsonWriter[T]): JsonWriter[Try[T]] = jsonWriter[Try[T]] {
    case Failure(f) => JsObject(
      "error" -> JsObject(
        "class" -> JsString(f.getClass.getSimpleName),
        "message" -> JsString(f.getMessage)
      )
    )
    case Success(s) => JsObject(
      "data" -> w.write(s)
    )
  }

  implicit val installFromRepositoryOutcomeWriter: JsonWriter[InstallFromRepositoryOutcome] = jsonWriter[InstallFromRepositoryOutcome] { ifo =>
    JsObject(
      "id" -> ifo.id.toJson,
      "outcome" -> ifo.outcome.toJson
    )
  }

  def seqWriter[T](implicit w: JsonWriter[T]): JsonWriter[Seq[T]] = jsonWriter[Seq[T]] { seq =>
    JsArray(seq.map(_.toJson): _*)
  }

  implicit val installFromRepositoryOutcomeSeqWriter: JsonWriter[Seq[InstallFromRepositoryOutcome]] = seqWriter(installFromRepositoryOutcomeWriter)

  implicit val pluginRepositorySeqWriter: JsonWriter[Seq[PluginsRepository]] = seqWriter(pluginRepositoryWriter)
}