package com.xebialabs.plugin.manager.service

import java.io.File

import com.typesafe.config.{Config, ConfigObject}
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.plugin.manager.metadata.{ArtifactId, PluginsMetadata}
import com.xebialabs.plugin.manager.repository.PluginsRepository
import com.xebialabs.plugin.manager.repository.nexus.{NexusPluginRepository, NexusRepositoryConfig, NexusServerConfig}
import com.xebialabs.plugin.manager.service.PluginService.defaultTimeout
import com.xebialabs.plugin.manager.{Plugin, PluginId, PluginManager}
import com.xebialabs.xlplatform.upgrade.RepositoryVersionService
import grizzled.slf4j.Logging
import javax.annotation.PreDestroy

import scala.collection.JavaConverters._
import scala.collection.mutable
import scala.concurrent.duration._
import scala.concurrent.{Await, ExecutionContext, Future}
import scala.language.postfixOps
import scala.util.{Failure, Try}


trait PluginService extends Logging {

  def pluginManager: PluginManager

  implicit lazy val ec: ExecutionContext = pluginManager.ec

  def repositories: mutable.Map[String, PluginsRepository]

  def addRepository(repository: PluginsRepository): Boolean

  def deleteRepository(name: String): Boolean


  def update(): Unit = {
    if (repositories.nonEmpty) {
      logger.info(s"Updating all (${repositories.size}) repositories...")
      Await.ready(Future.sequence(repositories.values.map(_.update())), defaultTimeout)
    }
  }

  def search(query: Option[String]): Map[ArtifactId, PluginsMetadata] =  {
    logger.debug(s"search($query)")
    Await.result(
      for {
        installed <- pluginManager.search(query)
        available <- Future.sequence(repositories.values.map(_.search(query))).map(_.flatten)
      } yield {
        available.foldLeft(installed) {
          case (acc, (id, pm1@PluginsMetadata(em1, vs1, hl1))) =>
            acc.get(id) match {
              case None =>
                acc + (id -> pm1)
              case Some(PluginsMetadata(em0, vs0, hl0)) =>
                acc.updated(id, PluginsMetadata(em0 orElse em1, vs0 ++ vs1, hl0 || hl1))
            }
        }
      },
      defaultTimeout
    )
  }

  def search(query: String): Map[ArtifactId, PluginsMetadata] = search(Option(query).filter(_.nonEmpty))

  def list(): Map[ArtifactId, PluginsMetadata] = search(None)

  def listInstalled(): Seq[PluginId] =
    pluginManager.listInstalled()

  def install(plugin: Plugin): Unit = {
    logger.info(s"Installing plugin ${plugin.id.id}...")
    pluginManager.install(plugin)
  }

  def installFromRepository(id: PluginId.Artifact): Try[Unit] =
    repositories.get(id.repository).map { repo =>
      Try {
        val plugin = Await.result(repo.get(id), defaultTimeout)
        install(plugin)
      }.recoverWith {
        case err =>
          logger.warn(err.getMessage)
          Failure(err)
      }
    }.getOrElse {
      Failure(new NotFoundException(s"Unknown plugin repository '${id.repository}"))
    }

  def uninstall(id: PluginId): Boolean = {
    logger.info(s"Uninstalling plugin $id")
    pluginManager.uninstall(id)
  }

  def getLogo(id: ArtifactId): Option[File] =
    for {
      repositoryId <- id.repository
      repo <- repositories.get(repositoryId)
      logo <- repo.getLogo(id)
    } yield logo

  def attachMetadata: (ArtifactId, Seq[PluginId]) => (ArtifactId, PluginsMetadata) = {
    case (artifactId, group) =>
      val versions = group.map(_.pluginVersion).toSet
      val pluginsMetadata = {
        for {
          repoId <- artifactId.repository
          repo <- repositories.get(repoId)
          pluginsMeta <- repo.getMetadata(artifactId)
        } yield pluginsMeta.copy(versions = versions)
      }.getOrElse {
        PluginsMetadata.empty(versions)
      }

      artifactId -> pluginsMetadata
  }

  def extend(data: Seq[PluginId]): Map[ArtifactId, PluginsMetadata] =
    data.groupBy(_.toArtifactId).map(attachMetadata.tupled)


  @PreDestroy
  def shutdown(): Unit = {
    logger.info("Shutting down LocalPluginManager...")
    Await.ready(
      Future.sequence(repositories.values.collect {
        case nexus: NexusPluginRepository => nexus.shutdown()
      }),
      repositories.size * defaultTimeout
    )
  }

}


object PluginService {
  // TODO: get this from configuration
  val defaultTimeout: Duration = 180 seconds

  def serversFromConfig(pluginsConfig: Config): Map[String, Try[NexusServerConfig]] =
    Try(pluginsConfig.getObject("servers")).map(
      _.asScala.collect {
        case (serverName, serverConfig) =>
          serverName -> NexusServerConfig.fromConfig(serverConfig.asInstanceOf[ConfigObject].toConfig)
      }.toMap
    ).getOrElse(Map.empty)

  def repositoriesFromConfig(pluginsConfig: Config, servers: Map[String, Try[NexusServerConfig]])
                            (implicit repositoryVersionService: RepositoryVersionService): Map[String, Try[NexusRepositoryConfig]] =
    Try(pluginsConfig.getObject("repositories")).map(
      _.asScala
        .map { case (name, repoConfig) =>
          name -> NexusRepositoryConfig.fromConfig(name)(repoConfig.asInstanceOf[ConfigObject].toConfig, servers)
        }.toMap
    ).getOrElse(Map.empty)

  def configuredRepositories(pluginsConfig: Config)
                            (implicit repositoryVersionService: RepositoryVersionService): List[Try[NexusPluginRepository]] =
    repositoriesFromConfig(pluginsConfig, serversFromConfig(pluginsConfig))
      .map { case (name, tryConfig) =>
        tryConfig.map(config => NexusPluginRepository.memCached(name, config))
      }.toList
}