package com.xebialabs.plugin.manager.metadata

import akka.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import com.xebialabs.plugin.manager.requireNonEmptyNoColumnsAndNoSpaces
import spray.json._

import scala.util.matching.Regex

sealed trait Version

object Version {
  case object PluginVersion extends Version
  case class Constant private(major: Int, minor: Option[Int], revision: Option[Int], extra: Option[String]) extends Version {
    require(major >= 0, "version major can't be negative")
    require(minor.fold(true)(_ >= 0), "version minor can't be negative")
    require(revision.isEmpty || (revision.nonEmpty && minor.nonEmpty), "version revision requires version minor")
    revision.foreach(r => require(r >= 0, "version revision can't be negative"))
    extra.foreach(requireNonEmptyNoColumnsAndNoSpaces(_, "extra"))
  }

  object Constant {
    val zero = Constant(0)

    val versionConstantPattern: Regex = """(\d+)(\.\d+)?(\.\d+)?(.*)?""".r

    def apply(major: Int): Constant = new Constant(major, None, None, None)

    def apply(major: Int, minor: Int) = new Constant(major, Some(minor), None, None)

    def apply(major: Int, minor: Int, revision: Int): Constant = new Constant(major, Some(minor), Some(revision), None)

    def apply(major: Int, minor: Int, revision: Int, extra: Option[String]): Constant =
      new Constant(major, Some(minor), Some(revision), extra.filter(_.nonEmpty))

    def apply(major: Int, minor: Option[Int] = None, revision: Option[Int] = None, extra: Option[String] = None): Constant = {
      new Constant(major, minor, revision, extra.filter(_.nonEmpty))
    }

    def fromString(version: String): Option[Constant] = version match {
      case versionConstantPattern(major, minor, revision, extra) =>
        Some(
          Constant(
            major = Integer.parseInt(major),
            minor = Option(minor).map(v => Integer.parseInt(v.tail)),
            revision = Option(revision).map(v => Integer.parseInt(v.tail)),
            extra = Option(extra).filter(_.nonEmpty)
          )
        )
      case _ => None
    }

    implicit class ConstantOps(val version: Constant) extends AnyVal {
      def id: String = {
        s"${version.major}" ++
          version.minor.fold("")(minor => s".$minor" ++ version.revision.fold("")(revision => s".$revision")) ++
          version.extra.getOrElse("")
      }
    }

    trait Protocol extends SprayJsonSupport with DefaultJsonProtocol {

      val versionConstantWriter: JsonWriter[Version.Constant] = v => v.id.toJson
      val versionConstantReader: JsonReader[Version.Constant] = jsonReader[Version.Constant] {
        case JsString(versionString) =>
          Constant.fromString(versionString) match {
            case None =>
              deserializationError(s"Cannot parse constant version string: '$versionString'")
            case Some(version) => version
          }
        case unknown =>
          deserializationError(s"Cannot parse constant version: '$unknown'")
      }
      implicit val versionConstantFormat: JsonFormat[Version.Constant] = jsonFormat(versionConstantReader, versionConstantWriter)
    }

  }

  trait Protocol extends SprayJsonSupport with DefaultJsonProtocol with Constant.Protocol {

    private val versionPattern: Regex = """(\d+)\.(\d+)\.(\d+)(.*)""".r

    val versionString: Version => String = {
      case PluginVersion => "$pluginVersion"
      case c: Constant => c.id
    }
    val versionWriter: JsonWriter[Version] = v => versionString(v).toJson
    val versionReader: JsonReader[Version] = jsonReader[Version] {
      case JsString("$pluginVersion") => PluginVersion
      case constant: JsString => versionConstantReader.read(constant)
      case unknown =>
        deserializationError(s"Cannot parse version: '$unknown'")
    }
    implicit val versionFormat: JsonFormat[Version] = jsonFormat(versionReader, versionWriter)
  }
}