package com.xebialabs.plugin.manager.rest.api

import java.io.{InputStream, OutputStream}
import java.util.Date

import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.plugin.manager.PluginId.LocalFile
import com.xebialabs.plugin.manager.metadata.ArtifactId
import com.xebialabs.plugin.manager.repository.PluginsRepository
import com.xebialabs.plugin.manager.service.PluginService
import com.xebialabs.plugin.manager.{Plugin, PluginId}
import javax.ws.rs._
import javax.ws.rs.core.MediaType.APPLICATION_JSON
import javax.ws.rs.core.{MediaType, Response, StreamingOutput}
import org.apache.commons.io.IOUtils
import org.jboss.resteasy.plugins.providers.multipart.MultipartFormDataInput
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller
import spray.json._

import scala.collection.JavaConverters._
import scala.concurrent.duration._
import scala.concurrent.{Await, ExecutionContext, Future}
import scala.language.postfixOps
import scala.reflect.io.File


@Path("/plugin-manager")
@Produces(Array(APPLICATION_JSON))
@Controller
class PluginResource @Autowired()(val pluginService: PluginService) extends AbstractSecuredResource
  with PluginManagerProtocol {

  implicit val ec: ExecutionContext = pluginService.pluginManager.ec

  @GET
  @Produces(Array(MediaType.APPLICATION_JSON))
  def list(): Response = {
    checkPermission(ADMIN)
    Response.ok(
      pluginService.extend(pluginService.listInstalled()).map(pluginsResult).toJson.compactPrint,
      MediaType.APPLICATION_JSON
    ).build()
  }

  @POST
  @Consumes(Array("multipart/form-data"))
  def install(input: MultipartFormDataInput, @QueryParam("pluginId") pluginId: String): Unit = {
    checkPermission(ADMIN)
    val uploadForm = input.getFormDataMap
    val inputParts = uploadForm.get("file")

    inputParts.forEach { part =>
      val is = part.getBody[InputStream](classOf[InputStream], null)
      pluginService.install(Plugin(LocalFile(pluginId), None, is))
    }
  }

  @DELETE
  @Path("/delete")
  def uninstall(@QueryParam("pluginId") pluginId: String): Unit = {
    checkPermission(ADMIN)
    PluginId.fromIdString(pluginId) match {
      case Some(id) =>
        pluginService.uninstall(id)
      case None =>
        throw new NotFoundException(s"Plugin not currently installed: $pluginId")
    }
  }

  @POST
  @Path("/delete")
  @Consumes(Array(MediaType.APPLICATION_JSON))
  def uninstall(pluginIds: java.util.List[String]): Unit = {
    checkPermission(ADMIN)
    for {
      id <- pluginIds.asScala
      pluginId <- PluginId.fromIdString(id)
    } yield {
      pluginService.uninstall(pluginId)
    }
  }

  @GET
  @Path("search/{query}")
  @Produces(Array(MediaType.APPLICATION_JSON))
  def search(@PathParam("query") query: String): Response = {
    checkPermission(ADMIN)
    Response.ok(
      pluginService.search(query).map(pluginsResult).toSeq.toJson.compactPrint,
      MediaType.APPLICATION_JSON
    ).build()
  }

  @GET
  @Path("repositories/")
  @Produces(Array(MediaType.APPLICATION_JSON))
  def listRepositories(): Response = {
    checkPermission(ADMIN)
    Response.ok(pluginService.repositories.values.toSeq.toJson.compactPrint, MediaType.APPLICATION_JSON).build()
  }

  @POST
  @Path("repositories/{repositoryId}/update")
  def updateRepository(@PathParam("repositoryId") repositoryId: String): Date = {
    checkPermission(ADMIN)
    withRepo(repositoryId)(_.update().map(_.toDate))
  }

  @GET
  @Path("repositories/{repositoryId}/list")
  @Produces(Array(MediaType.APPLICATION_JSON))
  def listRepository(@PathParam("repositoryId") repositoryId: String): Response = {
    checkPermission(ADMIN)
    Response.ok(
      withRepo(repositoryId)(_.list()).map(pluginsResult).toSeq.toJson.compactPrint,
      MediaType.APPLICATION_JSON
    ).build()
  }

  @GET
  @Path("repositories/{repositoryId}/search/{query}")
  @Produces(Array(MediaType.APPLICATION_JSON))
  def searchRepository(@PathParam("repositoryId") repositoryId: String, @PathParam("query") query: String): Response = {
    checkPermission(ADMIN)
    Response.ok(
      withRepo(repositoryId)(_.search(query)).map(pluginsResult).toSeq.toJson.compactPrint,
      MediaType.APPLICATION_JSON
    ).build()
  }

  @POST
  @Path("repositories/{repositoryId}/install/{groupId}/{artifactId}/{version}")
  @Produces(Array(MediaType.APPLICATION_JSON))
  def installFromRepository(@PathParam("repositoryId") repositoryId: String,
                            @PathParam("groupId") groupId: String,
                            @PathParam("artifactId") artifactId: String,
                            @PathParam("version") version: String): Response = {
    checkPermission(ADMIN)
    val id = PluginId.Artifact.apply(repositoryId, groupId, artifactId, version)
    pluginService.installFromRepository(id).map { _ =>
      Response.ok(id.toJson, MediaType.APPLICATION_JSON).build()
    }.fold(
      err => throw err,
      ok => ok
    )
  }

  @GET
  @Path("repositories/{repositoryId}/logo/{groupId}/{artifactId}")
  @Produces(Array("image/*"))
  def getLogo(@PathParam("repositoryId") repositoryId: String,
              @PathParam("groupId") groupId: String,
              @PathParam("artifactId") artifactId: String): Response = {
    checkPermission(ADMIN)
    val id = ArtifactId(groupId, artifactId, Some(repositoryId))
    pluginService.getLogo(id).map { logoFile =>
      val input = File(logoFile).inputStream()
      val stream = new StreamingOutput {
        override def write(output: OutputStream): Unit = output.write(IOUtils.toByteArray(input))
      }
      Response.ok(stream, MediaType.APPLICATION_OCTET_STREAM).build()
    }.getOrElse {
      throw new NotFoundException(s"Couldn't find logo for '$groupId:$artifactId' in repository '$repositoryId'")
    }
  }

  private def withRepo[T](repositoryId: String, duration: Duration = 30 seconds)(action: PluginsRepository => Future[T]): T = {
    withRepo0(repositoryId) { repository =>
      Await.result(action(repository), duration)
    }
  }

  private def withRepo0[T](repositoryId: String)(action: PluginsRepository => T): T = {
    pluginService.repositories.get(repositoryId) match {
      case None =>
        throw new NotFoundException(s"Unknown plugin repository '$repositoryId'")

      case Some(repository) =>
        action(repository)
    }
  }

}
