package com.xebialabs.plugin.manager.metadata

import akka.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import com.typesafe.config.Config
import com.xebialabs.plugin.manager.{LocalPluginManager, PluginId, requireNonEmptyNoColumnsAndNoSpaces}
import spray.json._

import scala.util.Try

case class ArtifactId(groupId: String, artifactId: String, repository: Option[String] = None) {
  requireNonEmptyNoColumnsAndNoSpaces(groupId, "groupId")
  requireNonEmptyNoColumnsAndNoSpaces(artifactId, "artifactId")
  repository.foreach(r => {
    requireNonEmptyNoColumnsAndNoSpaces(r, "repository")
    require(r != LocalPluginManager.name)
  })
}

object ArtifactId {
  implicit class ArtifactIdOps(val artifactId: ArtifactId) extends AnyVal {
    def id: String = s"${artifactId.repository.getOrElse(LocalPluginManager.name)}:${artifactId.groupId}:${artifactId.artifactId}"

    def toPluginId(version: Option[Version.Constant], classifier: Option[String] = None): Option[PluginId] = (version, artifactId.repository) match {
      case (any, None) if artifactId.groupId == PluginId.localGroupId =>
        Some(toLocal(any))
      case (Some(v), Some(repo)) if artifactId.groupId != PluginId.localGroupId && repo != LocalPluginManager.name =>
        Some(toArtifact(repo, v, classifier))
      case _ =>
        None
    }

    def toArtifact(repositoryName: String, version: Version.Constant, classifier: Option[String] = None): PluginId.Artifact =
        PluginId.Artifact.parsed(repositoryName, artifactId.groupId, artifactId.artifactId, version, classifier)

    def toLocal(version: Option[Version.Constant]): PluginId.LocalFile =
      PluginId.LocalFile.parsed(artifactId.artifactId, version)
  }

  def fromConfig(repository: Option[String] = None)(repositoryConfig: Config): Try[ArtifactId] = for {
    groupId <- Try(repositoryConfig.getString("group-id"))
    artifactId <- Try(repositoryConfig.getString("artifact-id"))
  } yield ArtifactId(groupId, artifactId, repository)

  trait Protocol extends SprayJsonSupport with DefaultJsonProtocol {

    val artifactIdReader: RootJsonReader[ArtifactId] = {
      case obj: JsObject =>
        obj.getFields("groupId", "artifactId") match {
          case Seq(JsString(groupId), JsString(artifactId)) =>
            ArtifactId(groupId, artifactId)
          case wrong =>
            deserializationError(s"Cannot parse artifactId object: $wrong")
        }
      case unknown =>
        deserializationError(s"Cannot parse artifactId object: $unknown")
    }
    val artifactIdWriter: RootJsonWriter[ArtifactId] = (id: ArtifactId) => JsObject(
      "groupId" -> id.groupId.toJson,
      "artifactId" -> id.artifactId.toJson
    )

    implicit val artifactIdFormat: RootJsonFormat[ArtifactId] = rootJsonFormat(artifactIdReader, artifactIdWriter)
  }
}