package com.xebialabs.plugin.manager.repository

import java.io.{ByteArrayInputStream, File}

import com.xebialabs.plugin.manager.metadata.{ArtifactId, ExtendedMetadata, PluginsMetadata}
import com.xebialabs.plugin.manager.repository.nexus.NexusPluginRepositoryException
import com.xebialabs.plugin.manager.repository.storage.PluginMetadataStorage
import com.xebialabs.plugin.manager.repository.storage.PluginMetadataStorageException.UnknownPlugin
import com.xebialabs.plugin.manager.{Plugin, PluginId}
import grizzled.slf4j.Logging
import org.joda.time.DateTime

import scala.concurrent.Future


abstract class CachedPluginRepository[R](toPluginId: R => Option[PluginId.Artifact]) extends PluginsRepository with Logging {
  protected var timestamp: Option[DateTime] = None

  override def lastUpdated: Option[DateTime] = timestamp

  def cache: PluginMetadataStorage

  def getFromRemote(id: PluginId.Artifact): Future[ByteArrayInputStream]

  def searchFromRemote(query: Option[String]): Future[Seq[R]]

  override def get(id: PluginId.Artifact): Future[Plugin] = cache.get(id) match {
    case None =>
      Future.failed(UnknownPlugin(name, id))

    case Some(PluginsMetadata(metadata, _, _)) =>
      getFromRemote(id).map(content => Plugin(id, metadata, content))
  }

  override def getLogo(id: ArtifactId): Option[File] = cache.getLogo(id)

  override def update(): Future[DateTime] = this.synchronized {
    logger.debug("update()")
    for {
      _ <- updateLogos()
      _ <- updateMetadata()
    } yield {
      timestamp = Some(DateTime.now)
      timestamp.get
    }
  }

  override def search(query: Option[String]): Future[Map[ArtifactId, PluginsMetadata]] =
    Future.successful(cache.search(query))

  override def getMetadata(id: ArtifactId): Option[PluginsMetadata] =
    cache.get(id)

  protected def updateMetadata(): Future[Unit] = {
    logger.trace("updateMetadata()")
    for {
      data <- searchFromRemote(None)
      metadata <- fetchMetadata.recoverWith {
        case _: NexusPluginRepositoryException =>
          Future.successful(Map.empty[ArtifactId, ExtendedMetadata])
      }
    } yield {
      val mapped = data.map(toPluginId).flatten
      cache.clear()
      cache.store(mapped, metadata)
    }
  }

  protected def updateLogos(): Future[Unit] = {
    logger.trace("updateLogos()")
    fetchLogos
      .map(cache.storeLogos)
      .recoverWith {
        case _: NexusPluginRepositoryException =>
          Future.successful(())
      }
  }

}
