package com.xebialabs.plugin.manager.repository.nexus

import akka.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import com.typesafe.config.{Config, ConfigException}
import com.xebialabs.plugin.manager.metadata.Version
import com.xebialabs.plugin.manager.repository.nexus.NexusRepositoryConfig.ListMethod.ByGroupId
import com.xebialabs.plugin.manager.repository.nexus.NexusRepositoryConfig.MetadataId
import com.xebialabs.plugin.manager.service.ProductConfig
import com.xebialabs.plugin.manager.{PluginId, requireNonEmptyNoColumnsAndNoSpaces}
import spray.json._

import scala.language.implicitConversions
import scala.util.{Failure, Try}


case class NexusRepositoryConfig(server: NexusServerConfig,
                                 productVersion: Version,
                                 repositoryId: String,
                                 metadataArtifactId: MetadataId,
                                 listMethod: NexusRepositoryConfig.ListMethod.ByGroupId) {
  lazy val metadataVersion: Version = productVersion.copy(extra = None)
}

object NexusRepositoryConfig {

  def byGroupId(server: NexusServerConfig,
                productVersion: Version,
                repositoryId: String,
                metadataId: MetadataId,
                groupId: String,
                packagingType: Option[String] = None,
                classifier: Option[String] = None): NexusRepositoryConfig = {
    NexusRepositoryConfig(server, productVersion, repositoryId, metadataId, ListMethod.ByGroupId(groupId, packagingType, classifier))
  }

  case class MetadataId(groupId: String, artifactId: String) {
    requireNonEmptyNoColumnsAndNoSpaces(groupId, "groupId")
    requireNonEmptyNoColumnsAndNoSpaces(artifactId, "artifactId")
  }

  object MetadataId {
    def fromConfig(config: Config): Try[MetadataId] =
      for {
        groupId <- Try(config.getString("group-id"))
        artifactId <- Try(config.getString("artifact-id"))
      } yield MetadataId(groupId, artifactId)

    trait Protocol extends SprayJsonSupport with DefaultJsonProtocol {
      implicit val metadataIdWriter: JsonWriter[MetadataId] = metadataId =>
        JsObject(
          "group-id" -> metadataId.groupId.toJson,
          "artifact-id" -> metadataId.artifactId.toJson
        )
    }

    implicit class MetadataIdOps(val metadataId: MetadataId) extends AnyRef {
      def id: String = s"${metadataId.groupId}:${metadataId.artifactId}"

      def toArtifact(repoName: String, version: Version, classifier: Option[String] = None): PluginId.Artifact =
        PluginId.Artifact.parsed(repoName, metadataId.groupId, metadataId.artifactId, version, classifier)
    }

  }

  sealed trait ListMethod

  object ListMethod {

    case class ByGroupId(groupId: String, packaging: Option[String], classifier: Option[String]) extends ListMethod {
      requireNonEmptyNoColumnsAndNoSpaces(groupId, "groupId")
    }

    object ByGroupId {
      def fromConfig(repositoryConfig: Config): Try[ByGroupId] = {
       Try(repositoryConfig.getString("nexus-group-id")).map { groupId =>
          val packagingType = Try(repositoryConfig.getString("nexus-packaging-type")).toOption
         ByGroupId(groupId, packagingType, None)
        }
      }
    }

    def fromConfig(repositoryConfig: Config): Try[ListMethod] = {
      Try(repositoryConfig.getString("repository-type")).flatMap {
        case "nexus-by-group-id" => ByGroupId.fromConfig(repositoryConfig)
        case unknownRepoType =>
          Failure(new ConfigException.WrongType(repositoryConfig.origin(), "repository-type", "'nexus-by-group-id'", unknownRepoType))
      }
    }
  }

  def fromConfig(name: String)(repositoryConfig: Config, servers: Map[String, Try[NexusServerConfig]])
                (implicit productConfig: ProductConfig): Try[NexusRepositoryConfig] = {
    for {
      _ <- Try(repositoryConfig.getBoolean("enabled")).filter(identity)
      server <- if (repositoryConfig.hasPath("server-ref")) {
        val serverRefName = repositoryConfig.getString("server-ref")
        servers.get(serverRefName) match {
          case None => Failure(new ConfigException.Missing(s"servers.$serverRefName"))
          case Some(serverConfig) => serverConfig
        }
      } else {
        NexusServerConfig.fromConfig(repositoryConfig.getConfig("server"))
      }
      listMethod <- ListMethod.ByGroupId.fromConfig(repositoryConfig)
      repositoryId <- Try(repositoryConfig.getString("nexus-repository-id"))
      metadataId <- Try(repositoryConfig.getConfig("nexus-metadata-artifact")).flatMap(MetadataId.fromConfig)
    } yield {
      NexusRepositoryConfig(server, productConfig.version, repositoryId, metadataId, listMethod)
    }
  }

  trait Protocol extends SprayJsonSupport with DefaultJsonProtocol
    with NexusServerConfig.Protocol
    with MetadataId.Protocol {

    implicit val nexusConfigWriter: RootJsonWriter[NexusRepositoryConfig] = config =>
      JsObject(
        "server" -> config.server.toJson,
        "repository-id" -> config.repositoryId.toJson,
        "nexus-metadata-artifact" -> config.metadataArtifactId.toJson,
        "repository-type" -> "nexus-by-group-id".toJson,
        "nexus-group-id" -> config.listMethod.groupId.toJson,
        "nexus-packaging-type" -> config.listMethod.packaging.toJson
      )
  }
}

