package com.xebialabs.plugin.manager.repository.nexus

import akka.http.scaladsl.model.StatusCode
import com.xebialabs.deployit.engine.spi.exception.{DeployitException, HttpResponseCodeResult}
import com.xebialabs.plugin.manager.PluginId
import com.xebialabs.plugin.manager.metadata.{ArtifactId, Version}

sealed abstract class NexusPluginRepositoryException(msg: String) extends DeployitException(msg)

object NexusPluginRepositoryException {

  @HttpResponseCodeResult(statusCode = 503)
  case class DownloadArtifactException(pluginId: PluginId.Artifact, statusCode: StatusCode, payload: String)
    extends NexusPluginRepositoryException(s"Artifact ${pluginId.id}: failed to download artifact: ${statusCode.intValue()}) $payload")

  @HttpResponseCodeResult(statusCode = 503)
  case class DownloadChecksumException(pluginId: PluginId.Artifact, statusCode: StatusCode, payload: String)
    extends NexusPluginRepositoryException(s"Artifact ${pluginId.id}: failed to download the SHA1 checksum: (${statusCode.intValue()}) $payload")

  @HttpResponseCodeResult(statusCode = 503)
  case class ChecksumMismatch(pluginId: PluginId.Artifact, found: String, expected: String)
    extends NexusPluginRepositoryException(s"Artifact ${pluginId.id}: SHA1 checksum mismatch: found: $found, expected: $expected")

  @HttpResponseCodeResult(statusCode = 503)
  case class ContentLengthMismatch(pluginId: PluginId.Artifact, found: Long, expected: Long)
    extends NexusPluginRepositoryException(s"Artifact ${pluginId.id}: content-length mismatch: found: $found, expected: $expected")

  @HttpResponseCodeResult(statusCode = 503)
  case class QueryFailed(query: Option[String], statusCode: StatusCode, payload: String)
    extends NexusPluginRepositoryException(query.fold("List query")(q => "Searching for '$q'") ++ s" failed: (${statusCode.intValue()}) $payload")

  @HttpResponseCodeResult(statusCode = 503)
  case class SearchMetadataFailed(artifactId: ArtifactId, packaging: String, classifier: Option[String], statusCode: StatusCode, payload: String)
    extends NexusPluginRepositoryException(
      s"Searching metadata version for ${artifactId.id}${classifier.fold("")("-"+ _)}.$packaging failed: (${statusCode.intValue()}) $payload"
    )

  @HttpResponseCodeResult(statusCode = 503)
  case class NoMetadataVersions(artifactId: ArtifactId, packaging: String, classifier: Option[String], pv: Version.Constant, versions: Seq[Version.Constant])
    extends NexusPluginRepositoryException(
      s"No metadata versions found matching ${pv.id}. Metadata artifact ${artifactId.id}${classifier.fold("")("-"+ _)}.$packaging. Available versions: ${versions.map(_.id).mkString(",")}"
    )
}
