package com.xebialabs.plugin.manager

import java.nio.file.{FileAlreadyExistsException, Files, Path, Paths}

import com.xebialabs.deployit.repository.ItemConflictException
import com.xebialabs.plugin.manager.LocalPluginManager.{listDir, pluginsDir}
import grizzled.slf4j.Logging

import scala.collection.JavaConverters._
import scala.concurrent.ExecutionContext

object LocalPluginManager {
  val name = "__local__"

  val pluginsDir: Path = Paths.get("plugins")


  def listDir(dir: Path): Stream[Path] =
    if (dir.toFile.isDirectory) {
      Files.list(dir).iterator().asScala.toStream
    } else {
      Stream.empty
    }

}

class LocalPluginManager(repositoryGroupId: Map[String, String]) extends PluginManager with Logging {
  def name: String = LocalPluginManager.name

  implicit val ec: ExecutionContext = ExecutionContext.global

  def ensurePluginsDirectoryExists(): Unit = {
    pluginsDir.resolve(name).toFile.mkdirs()
  }

  override def listInstalled(): Seq[PluginId] = {
    ensurePluginsDirectoryExists()
    listDir(pluginsDir).flatMap(listInstalledIn)
  }

  override def install(plugin: Plugin): Unit = {
    logger.debug(s"install(${plugin.id.id}, ${plugin.metadata})")
    ensurePluginsDirectoryExists()
    uninstallPreviousPlugin(plugin.id)
    val destDir = pluginsDir.resolve(plugin.id.path)
    destDir.toFile.mkdirs()
    val dest = destDir.resolve(plugin.id.filename)
    try {
      Files.copy(plugin.content, dest)
    } catch {
      case _: FileAlreadyExistsException =>
        throw new ItemConflictException("A plugin named '%s' is already installed.", plugin.id.id)
    } finally {
      plugin.content.close()
    }
  }

  override def uninstall(id: PluginId): Boolean = {
    logger.debug(s"uninstall(${id.id})")
    val file = pluginsDir.resolve(id.path).resolve(id.filename)
    Files.deleteIfExists(file)
  }

  private def uninstallPreviousPlugin(id: PluginId): Unit = {
    logger.debug(s"uninstallPreviousPlugin(${id.id})")
    listInstalled()
      .filter(_.matching(id))
      .foreach(uninstall)
  }

  protected def listInstalledIn(dir: Path): Seq[PluginId] =
    if (dir.getFileName.toString == name) {
      listLocal(dir)
    } else {
      listRepository(dir)
    }

  protected def listLocal(localDir: Path): Stream[PluginId] =
    listPluginDirectory(localDir) { filename =>
      PluginId.LocalFile.fromIdString(s"$name:${PluginId.localGroupId}:$filename")
    }

  protected def listRepository(repositoryDir: Path): Stream[PluginId] = {
    val repository = repositoryDir.getFileName.toString
    repositoryGroupId.get(repository).map { groupId =>
      listPluginDirectory(repositoryDir) { filename =>
        PluginId.Artifact.fromIdString(s"$repository:$groupId:$filename")
      }
    }.getOrElse(Stream.empty)
  }

  protected def listPluginDirectory(path: Path)(f: String => Option[PluginId]): Stream[PluginId] =
    listDir(path)
      .map(_.getFileName.toString)
      .filter(_.endsWith(".jar"))
      .flatMap(filename => f(filename))

}
