package com.xebialabs.plugin.manager.service

import com.xebialabs.deployit.engine.spi.exception.{DeployitException, HttpResponseCodeResult}
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.plugin.manager.config.PluginManagerProperties
import com.xebialabs.plugin.manager.metadata.{ArtifactId, PluginMetadata}
import com.xebialabs.plugin.manager.repository.PluginsRepository
import com.xebialabs.plugin.manager.repository.nexus.{NexusPluginRepository, NexusRepositoryConfig, NexusServerConfig}
import com.xebialabs.plugin.manager.rest.api.{PluginSource, PluginStatus}
import com.xebialabs.plugin.manager.rest.dto.PluginDto
import com.xebialabs.plugin.manager.service.PluginService.defaultTimeout
import com.xebialabs.plugin.manager.sql.DbPlugin
import com.xebialabs.plugin.manager.{Plugin, PluginId, PluginManager}
import grizzled.slf4j.Logging

import java.io.File
import javax.annotation.PreDestroy
import scala.collection.mutable
import scala.concurrent.duration._
import scala.concurrent.{Await, ExecutionContext, Future}
import scala.jdk.CollectionConverters._
import scala.language.postfixOps
import scala.util.{Failure, Success, Try}

trait PluginService extends Logging {

  def pluginManager: PluginManager

  implicit lazy val ec: ExecutionContext = pluginManager.ec

  def repositories: mutable.Map[String, PluginsRepository]

  def addRepository(repository: PluginsRepository): Boolean

  def deleteRepository(name: String): Boolean


  def update(): Unit = {
    if (repositories.nonEmpty) {
      Await.ready(Future.sequence(repositories.values.map(_.update())), defaultTimeout)
    }
  }

  def search(query: Option[String]): Map[ArtifactId, Option[PluginMetadata]] = {
    logger.debug(s"search($query)")
    Await.result(
      for {
        installed <- pluginManager.search(query)
        available <- Future.sequence(repositories.values.map(_.search(query))).map(_.flatten)
      } yield {
        available.foldLeft(installed.map(_ -> Option.empty[PluginMetadata]).toMap) {
          case (acc, (id, pm1)) =>
            acc.get(id) match {
              case None =>
                acc.updated(id, pm1)
              case Some(_) =>
                pm1.map(pm => acc.updated(id, Some(pm))).getOrElse(acc)
            }
        }
      },
      defaultTimeout
    )
  }

  def searchOfficial(query: String): Map[ArtifactId, Option[PluginMetadata]] = {
    Await.result(
      for {
        available <- Future.sequence(repositories.values.map(_.search(Some(query)))).map(_.flatten)
      } yield {
        available.toMap
      },
      defaultTimeout
    )
  }

  def search(query: String): Map[ArtifactId, Option[PluginMetadata]] = search(Option(query).filter(_.nonEmpty))

  def list(): Map[ArtifactId, Option[PluginMetadata]] = search(None)

  def listInstalled(): Seq[PluginId] =
    pluginManager.listInstalled()

  def install(plugin: Plugin): Unit = {
    logger.info(s"Installing plugin ${plugin.id.id}...")
    pluginManager.install(plugin)
    logger.info(s"Installation of plugin ${plugin.id.id} to the database is complete. " +
      s"System restart required for the plugin to be ready for usage.")
  }

  def installFromRepository(id: PluginId.Artifact): Try[Unit] =
    repositories.get(id.repository).map { repo =>
      Try {
        val plugin = Await.result(repo.get(id), defaultTimeout)
        install(plugin)
      }.recoverWith {
        case err =>
          logger.warn(err.getMessage)
          Failure(err)
      }
    }.getOrElse {
      Failure(new NotFoundException(s"Unknown plugin repository '${id.repository}"))
    }

  def uninstall(repositoryId: PluginSource.Value,
                groupId: String,
                artifactId: String,
                version: Option[String]): Try[Unit] = {
    logger.info(s"Uninstalling plugin $repositoryId:$groupId:$artifactId-$version")

    pluginManager.getPluginBy(repositoryId, groupId, artifactId, version) match {
      case Some(plugin) =>
        if (pluginManager.uninstall(plugin)) {
          pluginManager.revertPreviousVersionIfNecessary(plugin)
          Success()
        } else {
          Failure(PluginUninstallException(plugin, s"Only plugins in ${PluginStatus.READY_FOR_INSTALL} status can be uninstalled."))
        }
      case None => Failure(PluginNotFoundException(s"Plugin $repositoryId:$groupId:$artifactId-$version not found in the system."))
    }
  }

  def getLogo(repositoryId: String, artifactId: String): Option[File] =
    for {
      repo <- repositories.get(repositoryId)
      logo <- repo.getLogo(artifactId)
    } yield logo

  def attachMetadata(artifactId: ArtifactId): PluginDto = {
    val metadata = for {
      repoId <- artifactId.repository
      repo <- repositories.get(repoId)
      pluginsMeta <- repo.getMetadata(artifactId)
    } yield pluginsMeta
    PluginDto(artifactId, metadata)
  }

  def attachStatus: PluginDto => PluginDto = {
    val statuses: Map[String, PluginStatus.Value] = pluginManager.fetchPluginStatuses()
    pluginDto => pluginDto.copy(status = statuses.get(pluginDto.artifactId.artifactId))
  }

  def extend(data: Seq[PluginId]): Seq[PluginDto] = {
    data.map(_.toArtifactId).toSet.map(attachMetadata).map(attachStatus).toSeq
  }

  @PreDestroy
  def shutdown(): Unit = {
    logger.info("Shutting down LocalPluginManager...")
    Await.ready(
      Future.sequence(repositories.values.collect {
        case nexus: NexusPluginRepository => nexus.shutdown()
      }),
      repositories.size * defaultTimeout
    )
  }

}

object PluginService {
  // TODO: get this from configuration
  val defaultTimeout: Duration = 180 seconds

  def serversFromConfig(pluginsConfig: PluginManagerProperties): Map[String, Try[NexusServerConfig]] =
    Option(pluginsConfig.getServers) match {
      case Some(servers) => servers.asScala.collect(server =>
        server.getName -> NexusServerConfig.fromConfig(server)).toMap
      case None => Map.empty
    }

  def repositoriesFromConfig(pluginsConfig: PluginManagerProperties, servers: Map[String, Try[NexusServerConfig]])
                            (implicit productConfig: ProductConfig): Map[String, Try[NexusRepositoryConfig]] =
    Option(pluginsConfig.getRepositories) match {
      case Some(repositories) => repositories.asScala.map(repo =>
        repo.getName -> NexusRepositoryConfig.fromConfig(repo.getName)(repo, servers)).toMap
      case None => Map.empty
    }

  def configuredRepositories(pluginsConfig: PluginManagerProperties)
                            (implicit productConfig: ProductConfig): List[Try[NexusPluginRepository]] =
    repositoriesFromConfig(pluginsConfig, serversFromConfig(pluginsConfig))
      .map { case (name, tryConfig) =>
        tryConfig.map(config => NexusPluginRepository.memCached(name, config))
      }.toList
}

@HttpResponseCodeResult(statusCode = 501)
case class PluginUninstallException(plugin: DbPlugin, message: String) extends DeployitException(s"Unable to uninstall plugin $plugin. $message")

@HttpResponseCodeResult(statusCode = 404)
case class PluginNotFoundException(message: String) extends DeployitException(message)
