package com.xebialabs.plugin.manager.materializer

import com.xebialabs.plugin.manager.config.ConfigWrapper
import com.xebialabs.plugin.manager.config.ConfigWrapper.{GROUP_ID_XLD, GROUP_ID_XLR}
import com.xebialabs.plugin.manager.metadata.XLProduct
import com.xebialabs.plugin.manager.metadata.XLProduct.{XLDeploy, XLRelease}
import com.xebialabs.plugin.manager.rest.api.PluginSource.{LOCAL, XLD_OFFICIAL, XLR_OFFICIAL}
import com.xebialabs.plugin.manager.rest.api.PluginStatus.READY_FOR_INSTALL
import com.xebialabs.plugin.manager.rest.api.{PluginSource, PluginStatus}
import com.xebialabs.plugin.manager.sql.DbPlugin.calculateChecksum
import com.xebialabs.plugin.manager.sql.{DbPlugin, SqlPluginRepository}
import com.xebialabs.xlplatform.sugar.PathSugar.path2File
import grizzled.slf4j.Logging
import org.apache.commons.io.FileUtils

import java.io.File
import java.nio.file.{Files, Path, Paths}
import scala.language.postfixOps
import scala.util.{Failure, Success, Using}

class PluginsMaterializer(val pluginRepository: SqlPluginRepository, val product: String) extends Logging {

  val pluginsDir: Path = Paths.get("plugins")
  val xlProduct: XLProduct = XLProduct.fromString(product)
  initProductSpecifics(xlProduct)

  def materializePlugins(): Unit = {
    if (pluginsDir.exists && pluginRepository.pluginTablesExist) {
      try {
        val dbPluginsBySource = pluginRepository.getAllWithBytes.groupBy(_.source.toString)

        Using(Files.list(pluginsDir)) { stream =>
          stream.filter(_.isDirectory)
            .map(dir => PluginSource.withName(dir.getFileName.toString) ->
              dir.listFiles().filter(_.getName.endsWith(ConfigWrapper.extension)).map(filePath => FilePlugin(filePath.toPath)).toList)
            .forEach(fsPluginsBySourceEntry => {
              fsPluginsBySourceEntry._1 match {
                case XLD_OFFICIAL | XLR_OFFICIAL =>
                  processOfficial(fsPluginsBySourceEntry._2, dbPluginsBySource.getOrElse(fsPluginsBySourceEntry._1.toString, Seq.empty))
                case LOCAL =>
                  processLocal(fsPluginsBySourceEntry._2, dbPluginsBySource.getOrElse(fsPluginsBySourceEntry._1.toString, Seq.empty))
                case _ => error(s"Could not match plugins directory(source) value ${fsPluginsBySourceEntry._1.toString}, skipping..")
              }
            })
        } match {
          case Success(_) =>
            info(s"Successfully synchronized plugins between filesystem and database")
          case Failure(err) =>
            error(s"Error during plugin synchronization process with message: ${err.getMessage}")
        }
        pluginRepository.updateAllPluginsStatusTo(PluginStatus.INSTALLED)
      }
      catch {
        case e: Exception => error(s"Error during plugin materialization process with message: ${e.getMessage}")
      }
    }
  }

  /**
    * <p> Compares list of official plugins found on the filesystem and official plugins written in the database
    * <p> If plugin exists on the FS and not in the DB, then it gets written into the DB
    * <p> If plugin exists in the DB and not on the FS, then it gets written onto the FS
    * <p> If plugin exists both on FS and DB, then higher version wins (gets written into the DB or FS) and the other one gets cleaned up
    *
    * @param fsPlugins
    * list of official plugins found on the filesystem
    * @param dbPlugins
    * list of official plugins found in the database
    */
  def processOfficial(fsPlugins: List[FilePlugin], dbPlugins: Seq[DbPlugin]): Unit = {
    debug(s"Processing list of official plugins found on the filesystem $fsPlugins and in the database $dbPlugins")
    // cleanup DB
    dbPlugins.filter(_.higherVersionExistsIn(fsPlugins)).foreach(dbPlugin => cleanupFromDatabase(dbPlugin))

    // FS to DB
    fsPlugins.filter(fsPlugin => fsPlugin.doesntExistIn(dbPlugins) || fsPlugin.isHigherVersionThanAMatchIn(dbPlugins))
      .foreach(fsPlugin => writeOfficialToDatabase(fsPlugin))

    // cleanup FS
    val fsPluginsToCleanup = fsPlugins.filter(fsPlugin => fsPlugin.higherVersionExistsIn(dbPlugins) || fsPlugin.hasDifferentContentThanAVersionMatchIn(dbPlugins))
    fsPluginsToCleanup.foreach(fsPlugin => cleanupFromFilesystem(fsPlugin))
    val fsPluginsAfterCleanup = fsPlugins.filter(fsPlugin => !fsPluginsToCleanup.contains(fsPlugin))

    // DB to FS
    dbPlugins.filter(dbPlugin => dbPlugin.doesntExistIn(fsPluginsAfterCleanup) || dbPlugin.isHigherVersionThanAMatchIn(fsPluginsAfterCleanup) || dbPlugin.hasDifferentContentThanAVersionMatchIn(fsPluginsAfterCleanup))
      .foreach(dbPlugin => materializeToFilesystem(dbPlugin))
  }

  /**
    * <p> Compares list of local plugins found on the filesystem and local plugins written in the database
    * <p> If plugin exists on the FS and not in the DB, then it gets written into the DB
    * <p> If plugin exists in the DB and not on the FS, then it gets written onto the FS
    * <p> If plugin exists both on FS and DB, and checksums are not identical then DB plugin always gets materialized over the FS plugin
    *
    * @param fsPlugins
    * list of local plugins found on the filesystem
    * @param dbPlugins
    * list of local plugins found in the database
    */
  def processLocal(fsPlugins: List[FilePlugin], dbPlugins: Seq[DbPlugin]): Unit = {
    debug(s"Processing list of local plugins found on the filesystem $fsPlugins and in the database $dbPlugins")
    // FS to DB
    fsPlugins.filter(_.doesntExistIn(dbPlugins)).foreach(fsPlugin => writeLocalToDatabase(fsPlugin))

    // cleanup FS
    fsPlugins.filter(_.hasDifferentContentThanAMatchIn(dbPlugins)).foreach(fsPlugin => cleanupFromFilesystem(fsPlugin))

    // DB to FS
    dbPlugins.filter(dbPlugin => dbPlugin.doesntExistIn(fsPlugins) || dbPlugin.shouldReplaceAMatchIn(fsPlugins))
      .foreach(dbPlugin => materializeToFilesystem(dbPlugin))
  }

  def materializeToFilesystem(dbPlugin: DbPlugin): Unit = {
    info(s"Synchronizing plugin ${dbPlugin.name} from database to filesystem")
    val fileName = s"${dbPlugin.name}${if (dbPlugin.version.isEmpty) "" else "-" + dbPlugin.version.get}.${dbPlugin.extension}"
    val targetFile = new File(s"$pluginsDir${File.separator}${dbPlugin.source}${File.separator}$fileName")
    FileUtils.writeByteArrayToFile(targetFile, dbPlugin.bytes.get)
  }

  def writeOfficialToDatabase(fsPlugin: FilePlugin): Unit = {
    info(s"Synchronizing plugin ${fsPlugin.name} from filesystem to database")
    val groupId = if (xlProduct == XLProduct.XLDeploy) GROUP_ID_XLD else GROUP_ID_XLR
    val source = if (xlProduct == XLProduct.XLDeploy) XLD_OFFICIAL else XLR_OFFICIAL
    val bytes = Files.readAllBytes(fsPlugin.filePath)
    val dbPlugin = new DbPlugin(0, fsPlugin.name, Option(fsPlugin.version), fsPlugin.extension, groupId, READY_FOR_INSTALL, source, calculateChecksum(bytes), Some(0), Some(bytes))
    pluginRepository.insert(dbPlugin)
  }

  def writeLocalToDatabase(fsPlugin: FilePlugin): Unit = {
    info(s"Synchronizing plugin ${fsPlugin.name} from filesystem to database")
    val bytes = Files.readAllBytes(fsPlugin.filePath)
    val dbPlugin = new DbPlugin(0, fsPlugin.name, Option(fsPlugin.version), fsPlugin.extension, LOCAL.toString, READY_FOR_INSTALL, LOCAL, calculateChecksum(bytes), Some(0), Some(bytes))
    pluginRepository.insert(dbPlugin)
  }

  def cleanupFromFilesystem(fsPlugin: FilePlugin): Unit = {
    info(s"Removing plugin ${fsPlugin.name} from filesystem")
    FileUtils.forceDelete(fsPlugin.filePath)
  }

  def cleanupFromDatabase(dbPlugin: DbPlugin): Unit = {
    info(s"Removing plugin ${dbPlugin.name} from database")
    pluginRepository.delete(dbPlugin)
  }

  def initProductSpecifics(xlproduct: XLProduct): Unit = {
    xlproduct match {
      case XLDeploy => {
        ConfigWrapper.extension = ConfigWrapper.EXTENSION_XLDP
        ConfigWrapper.groupId = ConfigWrapper.GROUP_ID_XLD
      }
      case XLRelease => {
        ConfigWrapper.extension = ConfigWrapper.EXTENSION_JAR
        ConfigWrapper.groupId = ConfigWrapper.GROUP_ID_XLR
      }
      case other => throw new IllegalArgumentException(s"Unknown product name: '$other'")
    }
  }

}
