package com.xebialabs.plugin.manager

import java.nio.file.{FileAlreadyExistsException, Files, Path, Paths}
import com.xebialabs.xlplatform.utils.ResourceManagement.using
import com.xebialabs.deployit.repository.ItemConflictException
import com.xebialabs.plugin.manager.LocalPluginManager.{listDir, pluginsDir}
import com.xebialabs.plugin.manager.config.ConfigWrapper
import grizzled.slf4j.Logging

import scala.concurrent.ExecutionContext
import scala.jdk.CollectionConverters._

object LocalPluginManager {
  val name = "__local__"

  val pluginsDir: Path = Paths.get("plugins")

  def listDir(dir: Path): LazyList[Path] =
    if (dir.toFile.isDirectory) {
      using(Files.newDirectoryStream(dir)) { list =>
        list.asScala.to(LazyList).force
      }
    } else LazyList.empty

}

class LocalPluginManager(repositoryGroupId: Map[String, String]) extends PluginManager with Logging {
  def name: String = LocalPluginManager.name

  implicit val ec: ExecutionContext = ExecutionContext.global

  def ensurePluginsDirectoryExists(): Unit = {
    pluginsDir.resolve(name).toFile.mkdirs()
  }

  override def listInstalled(): Seq[PluginId] = {
    ensurePluginsDirectoryExists()
    listDir(pluginsDir).flatMap(listInstalledIn)
  }

  override def install(plugin: Plugin): Unit = {
    logger.debug(s"install(${plugin.id.id}, ${plugin.metadata})")
    ensurePluginsDirectoryExists()
    uninstallPreviousPlugin(plugin.id)
    val destDir = pluginsDir.resolve(plugin.id.path)
    destDir.toFile.mkdirs()
    val dest = destDir.resolve(plugin.id.filename)
    try {
      Files.copy(plugin.content, dest)
    } catch {
      case _: FileAlreadyExistsException =>
        throw new ItemConflictException("A plugin named '%s' is already installed.", plugin.id.id)
    } finally {
      plugin.content.close()
    }
  }

  override def uninstall(id: PluginId): Boolean = {
    logger.debug(s"uninstall(${id.id})")
    val file = pluginsDir.resolve(id.path).resolve(id.filename)
    Files.deleteIfExists(file)
  }

  private def uninstallPreviousPlugin(id: PluginId): Unit = {
    logger.debug(s"uninstallPreviousPlugin(${id.id})")
    listInstalled()
      .filter(_.matching(id))
      .foreach(uninstall)
  }

  protected def listInstalledIn(dir: Path): Seq[PluginId] =
    if (dir.getFileName.toString == name) {
      listLocal(dir)
    } else {
      listRepository(dir)
    }

  protected def listLocal(localDir: Path): LazyList[PluginId] =
    listPluginDirectory(localDir) { filename =>
      PluginId.LocalFile.fromIdString(s"$name:${PluginId.localGroupId}:$filename")
    }

  protected def listRepository(repositoryDir: Path): LazyList[PluginId] = {
    val repository = repositoryDir.getFileName.toString
    repositoryGroupId.get(repository).map { groupId =>
      listPluginDirectory(repositoryDir) { filename =>
        PluginId.Artifact.fromIdString(s"$repository:$groupId:$filename")
      }
    }.getOrElse(LazyList.empty)
  }

  protected def listPluginDirectory(path: Path)(f: String => Option[PluginId]): LazyList[PluginId] =
    listDir(path)
      .map(_.getFileName.toString)
      .filter(_.endsWith(s".${ConfigWrapper.extension}"))
      .flatMap(f)
}
