package com.xebialabs.plugin.manager.repository.nexus

import java.net.InetSocketAddress

import akka.actor.ActorSystem
import akka.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import akka.http.scaladsl.model.headers.{BasicHttpCredentials, HttpCredentials}
import akka.http.scaladsl.model.{HttpRequest, HttpResponse, Uri}
import akka.http.scaladsl.settings.ConnectionPoolSettings
import akka.http.scaladsl.{ClientTransport, Http}
import com.typesafe.config.Config
import com.xebialabs.plugin.manager.repository.nexus.NexusServerConfigHelper.{getCredentials, getProxyTransport}
import spray.json._

import scala.concurrent.Future
import scala.util.{Success, Try}

case class NexusServerConfig(uri: Uri, credentials: HttpCredentials, transport: ClientTransport = ClientTransport.TCP) {
  def serviceUri: Uri = uri.withPath(uri.path / "service" / "local")

  def settings(implicit system: ActorSystem): ConnectionPoolSettings = ConnectionPoolSettings(system).withTransport(transport)

  def httpRequest(req: HttpRequest)(implicit system: ActorSystem): Future[HttpResponse] =
    Http().singleRequest(req, settings = settings)
}

object NexusServerConfig {
  def apply(uri: Uri, credentials: HttpCredentials, proxyUri: Uri, proxyCredentialsOpt: Option[HttpCredentials]): NexusServerConfig = {
    val proxyAddress = new InetSocketAddress(proxyUri.authority.host.address(), proxyUri.authority.port)
    val transport = proxyCredentialsOpt match {
      case None => ClientTransport.httpsProxy(proxyAddress)
      case Some(proxyCredentials) => ClientTransport.httpsProxy(proxyAddress, proxyCredentials)
    }
    NexusServerConfig(uri, credentials, transport)
  }

  def fromConfig(serverConfig: Config): Try[NexusServerConfig] = for {
    _ <- Try(serverConfig.getString("server-type")).filter(_ == "nexus")
    uri <- Try(serverConfig.getString("url")).map(Uri.apply)
    credentials <- getCredentials(serverConfig).flatMap(identity)
    server <- getProxyTransport(serverConfig).fold(
      _ => Success(new NexusServerConfig(uri, credentials, transport = ClientTransport.TCP)),
      t => t.map(proxyTransport => new NexusServerConfig(uri, credentials, transport = proxyTransport))
    )
  } yield server

  trait Protocol extends SprayJsonSupport with DefaultJsonProtocol {
    implicit val nexusServerConfigWriter: RootJsonWriter[NexusServerConfig] = {
      case NexusServerConfig(uri, _, _) =>
        JsObject(
          "server-type" -> "nexus".toJson,
          "url" -> uri.toString.toJson
        )
    }
  }
}

object NexusServerConfigHelper {
  def getCredentials(config: Config): Try[Try[HttpCredentials]] = {
    Try(config.getConfig("credentials")).map { credentialsConfig =>
      for {
        username <- Try(credentialsConfig.getString("username"))
        password <- Try(credentialsConfig.getString("password"))
      } yield new BasicHttpCredentials(username, password)
    }
  }

  def getAddress(config: Config): Try[InetSocketAddress] = for {
    host <- Try(config.getString("host"))
    port <- Try(config.getInt("port"))
  } yield new InetSocketAddress(host, port)

  def getProxyTransport(config: Config): Try[Try[ClientTransport]] = for {
    proxyConfig <- Try(config.getConfig("proxy"))
  } yield {
    getAddress(proxyConfig).flatMap { proxyAddress =>
      getCredentials(proxyConfig).fold(
        _ => Success(ClientTransport.httpsProxy(proxyAddress)),
        proxyCredentials => proxyCredentials.map(ClientTransport.httpsProxy(proxyAddress, _))
      )
    }
  }
}
