package com.xebialabs.license;

import java.util.*;

import org.joda.time.LocalDate;

import static com.xebialabs.license.LicenseProperty.PRODUCT;

public abstract class License {

    public static final String PRODUCT_XL_DEPLOY = "XL Deploy";
    public static final String PRODUCT_OLD_DEPLOYIT = "Deployit";
    public static final String PRODUCT_XL_RELEASE = "XL Release";
    public static final String PRODUCT_XL_TESTVIEW = "XL TestView";

    public static final Set<String> PRODUCTS = new HashSet<>(Arrays.asList(PRODUCT_XL_DEPLOY, PRODUCT_XL_RELEASE, PRODUCT_OLD_DEPLOYIT, PRODUCT_XL_TESTVIEW));

    public static final int MAX_TRIAL_LICENSE_VALIDITY = 90;

    private LicensePropertyMap values;

    protected License(LicensePropertyMap values) throws LicenseViolationException {
        this.values = values;
        validateLicenseFormat();
    }

    public abstract List<LicenseProperty> getLicenseProperties();

    public abstract List<LicenseProperty> getRequiredProperties();

    public abstract int getLicenseVersion();

    protected void validateProperties() throws MissingLicensePropertyException {
        validateRequired();
    }

    private void validateRequired() throws MissingLicensePropertyException {
        for (LicenseProperty licenseProperty : getRequiredProperties()) {
            if (!hasLicenseProperty(licenseProperty)) {
                throw new MissingLicensePropertyException(emptyErrorMessage(licenseProperty));
            }
        }
    }

    private void validateProduct() throws LicenseProductException {
        String product = values.get(PRODUCT);
        if(!PRODUCTS.contains(product)) {
            throw new LicenseProductException(String.format("product should be one of: %s got '%s'", PRODUCTS, product));
        }
    }

    public void validateLicenseFormat() throws LicenseViolationException {
        validateProperties();
        validateProduct();
        validateTrialDate();
    }

    private void validateTrialDate() {
        if (this.isAtLeastVersion(LicenseVersion3.VERSION)) {
            if (values.getAsString(LicenseProperty.EDITION).equals(LicenseVersion3.Edition.Trial.name())) {
                final LocalDate expireDate = values.get(LicenseProperty.EXPIRES_AFTER);
                if (expireDate == null) {
                    throw new LicensePeriodExpiredException("Trial licenses require an expire date");
                }
                final LocalDate maxExpireDate = LocalDate.now().plusDays(MAX_TRIAL_LICENSE_VALIDITY);
                if (expireDate.isAfter(maxExpireDate)) {
                    throw new LicensePeriodExpiredException(String.format("Trial licenses cannot be valid for more than %s days. This license is valid until %s",
                            MAX_TRIAL_LICENSE_VALIDITY, expireDate));

                }
            }
        }

    }

    public boolean isDateExpired() {
        return isDateExpired(LocalDate.now());
    }

    public boolean isDateExpired(LocalDate now) {
        LocalDate localDateValue = getLocalDateValue(LicenseProperty.EXPIRES_AFTER);
        return localDateValue != null && now.isAfter(localDateValue);
    }

    public String getStringValue(LicenseProperty key) {
        return values.getAsString(key);
    }

    public boolean isDummyLicense() {
        return false;
    }

    public LocalDate getLocalDateValue(LicenseProperty key) {
        return values.get(key);
    }

    public Map<String, Integer> getMapValue(LicenseProperty key) {
        return values.get(key);
    }

    public List<String> getListValue(LicenseProperty property) {
        return new ArrayList<>(values.get(property));
    }

    public boolean hasLicenseProperty(LicenseProperty key) {
        return values.containsKey(key);
    }

    private String emptyErrorMessage(LicenseProperty property) {
        return String.format("'%s' cannot be empty", property.getName().toLowerCase());
    }

    private String formatAll(String format) {
        StringBuilder collector = new StringBuilder();
        for (LicenseProperty licenseProperty : getLicenseProperties()) {
            values.format(collector, format, licenseProperty);
        }
        return collector.toString();
    }

    public String toLicenseContent() {
        return formatAll("%s: %s\n");
    }

    @Override
    public String toString() {
        return formatAll("%-" + String.valueOf(LicenseProperty.getLongerNameLength()) + "s: %s\n");
    }

    public boolean isAtLeastVersion(int minimumLicenseVersion) {
        return getLicenseVersion() >= minimumLicenseVersion;
    }
}
