/**
 * Copyright © 2014-2016 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xlplatform.endpoints;

import java.util.HashMap;
import java.util.Map;

import com.xebialabs.xlplatform.documentation.PublicApiRef;

/**
 * Response object which is always available as {@code response} inside Jython scripts which implement REST endpoints in XL extensions.
 *
 * The content of HTTP response returned to the client will be the following:
 * <pre>
 * {@code
 *  {
 *      entity: "object, or array",
 *      stdout: "Text which has been sent to standard output during the execution",
 *      stderr: "Text which has been sent to standard error during the execution",
 *      exception: "Textual representation of exception that has been thrown during script execution"
 *
 *  }
 * }
 * </pre>
 */
@PublicApiRef
public class JythonResponse {

    private Object entity = null;

    private Integer statusCode = null;

    private Map<String, String> headers = new HashMap<String, String>();

    public Object getEntity() {
        return entity;
    }

    public Integer getStatusCode() {
        return statusCode;
    }

    public Map<String, String> getHeaders() {
        return headers;
    }

    /**
     * Sets body of the response which will be serialized into JSON and returned to the client.
     * {@code response.entity=[{"foo": 42, "bar": "some value"}, 34]}
     * Supported types are:
     * <ul>
     *     <li>Standard JSON data types: {@code Number}, {@code String}, {@code Boolean}, {@code Array*}, {@code Dictionary*}</li>
     *     <li>XL configuration items: ConfigurationItem</li>
     * </ul>
     * * {@code Array} and {@code Dictionary} can recursively contain aforementioned types as values of their elements.
     */
    public void setEntity(final Object entity) {
        this.entity = entity;
    }

    /**
     * Sets an HTTP status code for the response.
     * Normally this HTTP status code will be returned to the client. E.g.: {@code response.statusCode = 307}.
     * If this value is not set and script has been executed successfully, client will receive {@code 200}.
     * If script throws exception which is not caught, this value will be ignored and {@code 500} will be returned.
     */
    public void setStatusCode(final Integer statusCode) {
        this.statusCode = statusCode;
    }


    /**
     * Sets response headers. E.g.: {@code response.headers = {'Location': 'http://google.com'}}
     */
    public void setHeaders(final Map<String, String> headers) {
        this.headers = headers;
    }
}
