package com.xebialabs.xlplatform.cluster.membership

import akka.actor.{ActorLogging, ActorRef, Address, FSM, LoggingFSM, Props}
import akka.cluster.Cluster
import akka.pattern._
import com.xebialabs.xlplatform.cluster.membership.storage.ClusterMembershipManagement
import com.xebialabs.xlplatform.cluster.membership.storage.ClusterMembershipManagement.{Data, Failure, Seed, Success}

import scala.concurrent.ExecutionContextExecutor
import scala.concurrent.duration.FiniteDuration

class ClusterDiscoveryActor(cluster: Cluster, membershipMgmt: ClusterMembershipManagement, heartbeatInterval: FiniteDuration)
  extends LoggingFSM[ClusterDiscoveryActor.State, ClusterDiscoveryActor.Data] with ActorLogging {

  import ClusterDiscoveryActor._

  private implicit val executor: ExecutionContextExecutor = context.system.dispatcher

  when(Initial) {
    case Event(Start(stateManagerActor), _) =>
      membershipMgmt.createManagementInfra() pipeTo self
      stay() using MembershipRegistry(stateManagerActor)
    case Event(Failure(e), _) =>
      log.error(s"Could not create membership DB infra... trying registration anyway. Errormsg: $e")
      goto(Register)
    case Event(Success, _) =>
      goto(Register)
  }

  onTransition {
    case (_, Register) =>
      log.info("Starting node registration")
      val selfAddress: Address = cluster.selfAddress
      membershipMgmt.deregisterSeed(selfAddress).andThen {
        case util.Success(Success) => membershipMgmt.registerSelf(selfAddress)
      } pipeTo self
  }

  when(Register) {
    case Event(Success, _) =>
      goto(Join)
    case Event(Failure(e), _) =>
      log.error(s"Failed to (re-)register in the DB, so now stopping this node. Errormsg: $e")
      context.system.terminate()
      stop(FSM.Failure(e))
  }

  onTransition {
    case (_, Join) =>
      log.info("Listing seeds of the cluster from the DB")
      membershipMgmt.listActiveSeeds(cluster).map(data => {
        data match {
          case Data(seeds: List[Seed]) =>
            seeds.foreach(seed => {
              log.info("Found Seed {}", seed)
            })
          case _ => ()
        }
        data
      }) pipeTo self
  }

  when(Join) {
    case Event(Data(seeds: Seq[_]), MembershipRegistry(clusterStateManager)) =>
      log.info(s"Joining the cluster of: $seeds")
      val otherSeeds = seeds.map(_.asInstanceOf[Seed].address).filter(_ != cluster.selfAddress).toList
      if (otherSeeds.isEmpty) {
        log.info("Joining an empty cluster")
        cluster.join(cluster.selfAddress)
      } else {
        log.info(s"Joining a pre-seeded cluster at $otherSeeds")
        cluster.joinSeedNodes(otherSeeds)
      }
      context.actorOf(ClusterMembershipManagingActor.props(clusterStateManager, membershipMgmt))
      goto(Beating)
    case Event(Failure(e), _) =>
      log.error(s"Could not read seeds from DB - exiting. Errmsg: $e")
      context.system.terminate()
      stop(FSM.Failure(e))
  }

  onTransition {
    case (_, Beating) =>
      log.info("Starting heartbeat timer")
      startTimerAtFixedRate("heartbeat", Heartbeat, heartbeatInterval)
  }

  when(Beating) {
    case Event(Heartbeat, _) =>
      log.debug("Our heart is still beating...")
      membershipMgmt.heartbeat(cluster.selfAddress) pipeTo self
      stay()
    case Event(Success, _) =>
      stay()
    case Event(Failure(cause), _) =>
      log.warning(s"Heartbeat could not be written to the DB: $cause")
      stay()
  }

  onTermination {
    case StopEvent(reason, _, _) =>
      log.warning(s"Going down because of $reason - removing myself from the membership DB")
      membershipMgmt.deregisterSeed(cluster.selfAddress)
  }

  whenUnhandled {
    case Event(msg, _) =>
      log.warning(s"unhandled message $msg")
      stay()
  }

  startWith(Initial, Empty)
  initialize()
}

object ClusterDiscoveryActor {

  def props(cluster: Cluster, membershipMgmt: ClusterMembershipManagement, heartbeatInterval: FiniteDuration) =
    Props(new ClusterDiscoveryActor(cluster, membershipMgmt, heartbeatInterval))

  /*
   * FSM States
   */
  sealed trait State

  case object Initial extends State

  case object Register extends State

  case object Join extends State

  case object Beating extends State

  /**
    * FSM Data = known cluster nodes
    */
  sealed trait Data

  case object Empty extends Data

  case class MembershipRegistry(actorRef: ActorRef) extends Data

  /**
    * Message that triggers actor's initialization
    */
  case class Start(stateManagerActor: ActorRef)

  case object Heartbeat

}

