package com.xebialabs.xlplatform.cluster.membership

import akka.actor.{ActorRef, Address, LoggingFSM, Props}
import akka.cluster.Cluster
import akka.pattern._
import com.xebialabs.xlplatform.cluster.membership.storage.ClusterMembershipManagement
import com.xebialabs.xlplatform.cluster.membership.storage.ClusterMembershipManagement.{Data, Failure, Seed, Success}

import scala.concurrent.duration.FiniteDuration

class ClusterDiscoveryActor(cluster: Cluster, membershipMgmt: ClusterMembershipManagement, heartbeatInterval: FiniteDuration)
  extends LoggingFSM[ClusterDiscoveryActor.State, ClusterDiscoveryActor.Data] {

  import ClusterDiscoveryActor._

  private implicit val executor = context.system.dispatcher

  when(Initial) {
    case Event(Start(stateManagerActor), _) =>
      membershipMgmt.createManagementInfra() pipeTo self
      stay() using MembershipRegistry(stateManagerActor)
    case Event(Success, _) =>
      goto(Register)
  }

  onTransition {
    case (_, Register) =>
      log.info("Starting node registration")
      val selfAddress: Address = cluster.selfAddress
      membershipMgmt.deregisterSeed(selfAddress).andThen { case util.Success(Success) => membershipMgmt.registerSelf(selfAddress) } pipeTo self
  }

  when(Register) {
    case Event(Success, _) =>
      goto(Join)
    case Event(Failure(msg), _) =>
      log.error(msg)
      context.stop(self)
      stay()
  }

  onTransition {
    case (_, Join) =>
      log.info("Listing seeds of the cluster from the DB")
      membershipMgmt.listActiveSeeds(cluster) pipeTo self
  }

  when(Join) {
    case Event(Data(seeds: Seq[_]), MembershipRegistry(clusterStateManager)) =>
      log.info(s"Joining the cluster of: $seeds")
      val otherSeeds = seeds.map(_.asInstanceOf[Seed]).filterNot {
        _.address == cluster.selfAddress
      }
      if (otherSeeds.isEmpty) {
        log.info("Joining an empty cluster")
        cluster.join(cluster.selfAddress)
      } else {
        log.info(s"Joining a pre-seeded cluster at $otherSeeds")
        cluster.joinSeedNodes(otherSeeds.map(_.address).toList)
      }
      context.actorOf(ClusterMembershipManagingActor.props(clusterStateManager, membershipMgmt))
      goto(Beating)
  }

  onTransition {
    case (_, Beating) =>
      log.info("Starting heartbeat timer")
      setTimer("heartbeat", Heartbeat, heartbeatInterval, repeat = true)
  }

  when(Beating) {
    case Event(Heartbeat, _) =>
      log.debug("Our heart is still beating...")
      membershipMgmt.heartbeat(cluster.selfAddress) pipeTo self
      stay()
    case Event(Success, _) =>
      stay()
  }

  whenUnhandled {
    case Event(msg, _) ⇒
      log.warning(s"unhandled message $msg")
      stay()
  }

  startWith(Initial, Empty)
  initialize()
}

object ClusterDiscoveryActor {

  def props(cluster: Cluster, membershipMgmt: ClusterMembershipManagement, heartbeatInterval: FiniteDuration) =
    Props(classOf[ClusterDiscoveryActor], cluster, membershipMgmt, heartbeatInterval)

  /*
   * FSM States
   */
  sealed trait State

  case object Initial extends State

  case object Register extends State

  case object Join extends State

  case object Beating extends State

  /**
    * FSM Data = known cluster nodes
    */
  sealed trait Data

  case object Empty extends Data

  case class MembershipRegistry(actorRef: ActorRef) extends Data

  /**
    * Message that triggers actor's initialization
    */
  case class Start(stateManagerActor: ActorRef)

  case object Heartbeat

}

