package com.xebialabs.xlplatform.cluster.membership.storage

import java.time.Instant

import akka.actor.Address
import akka.cluster.Cluster
import com.xebialabs.xlplatform.cluster.DataSourceConfig
import com.xebialabs.xlplatform.cluster.membership.storage.ClusterMembershipManagement.Result

import scala.concurrent.{ExecutionContext, Future}
import scala.concurrent.duration.FiniteDuration

object ClusterMembershipManagement {

  sealed trait Result

  case object Success extends Result

  case class Data[+A](data: A) extends Result

  case class Failure[+A](reason: String) extends Result

  case class Seed(address: Address, ttl: Instant)

  val Oracle = "jdbc:oracle.*".r
  val MySQL = "jdbc:mysql.*".r
  val PostgreSQL = "jdbc:postgresql.*".r
  val DB2 = "jdbc:db2.*".r
  val MSSQL = "jdbc:sqlserver.*".r
  val H2 = "jdbc:h2.*".r
  val Derby = "jdbc:derby.*".r

  def selectDialect(dsConfig: DataSourceConfig, ttl: FiniteDuration, failFast: Boolean = true) = {
    val url: String = dsConfig.url
    url.toLowerCase match {
      case Oracle() => new ClusterMembershipOracleDialect(dsConfig, ttl, failFast)
      case MySQL() => new ClusterMembershipMySQLDialect(dsConfig, ttl, failFast)
      case PostgreSQL() => new ClusterMembershipPostgreSQLDialect(dsConfig, ttl, failFast)
      case DB2() => new ClusterMembershipDB2Dialect(dsConfig, ttl, failFast)
      case MSSQL() => new ClusterMembershipMSSQLDialect(dsConfig, ttl, failFast)
      case H2() => new ClusterMembershipH2Dialect(dsConfig, ttl, failFast)
      case Derby() => new ClusterMembershipDerbyDialect(dsConfig, ttl, failFast)
      case _ => throw new IllegalArgumentException(s"Could not determine the correct dialect for JDBC url: $url")
    }
  }

}

trait ClusterMembershipManagement {

  def createManagementInfra()(implicit executionContext: ExecutionContext): Future[Result]

  def registerSelf(self: Address)(implicit executionContext: ExecutionContext): Future[Result]

  def deregisterSeed(seed: Address)(implicit executionContext: ExecutionContext): Future[Result]

  def heartbeat(self: Address)(implicit executionContext: ExecutionContext): Future[Result]

  def listActiveSeeds(cluster: Cluster)(implicit executionContext: ExecutionContext): Future[Result]
}