package com.xebialabs.xlplatform.cluster.membership

import akka.actor.{Actor, ActorRef, Props}
import akka.cluster.ClusterEvent._
import akka.cluster.MemberStatus.Up
import akka.cluster.{Cluster, Member}
import com.xebialabs.xlplatform.cluster.membership.ClusterStateMessages.{ClusterMemberAdded, ClusterMemberLeft, Initialize, SelfAddedToCluster}
import com.xebialabs.xlplatform.cluster.membership.storage.ClusterMembershipManagement
import grizzled.slf4j.Logging


object ClusterStateMessages {

  sealed trait ClusterStateEvent

  case class SelfAddedToCluster(member: Member) extends ClusterStateEvent

  case object Initialize extends ClusterStateEvent

  case class ClusterMemberAdded(member: Member) extends ClusterStateEvent

  case class ClusterMemberLeft(member: Member) extends ClusterStateEvent

}

object ClusterMembershipManagingActor {
  def props(clusterStateManagingActor: ActorRef, clusterMembershipDialect: ClusterMembershipManagement) =
    Props(classOf[ClusterMembershipManagingActor], clusterStateManagingActor, clusterMembershipDialect)
}

/**
  * This actor listens to the cluster messages and updates the clusterStateManagingActor.
  * If this actor sees that this node has become Cluster Leader, it will also remove seed nodes from the member table in the persistent store.
  *
  * @param clusterStateManagingActor
  * @param membershipMgmt
  */
class ClusterMembershipManagingActor(clusterStateManagingActor: ActorRef, membershipMgmt: ClusterMembershipManagement) extends Actor with Logging {

  import context._

  lazy val cluster = Cluster(system)

  lazy val me = cluster.selfUniqueAddress.address

  @scala.throws[Exception](classOf[Exception])
  override def preStart(): Unit = {
    super.preStart()
    cluster.subscribe(self, InitialStateAsSnapshot, classOf[MemberUp], classOf[MemberRemoved])
  }

  @scala.throws[Exception](classOf[Exception])
  override def postStop(): Unit = {
    cluster.unsubscribe(self)
    super.postStop()
  }

  def deregisterSeedNode(member: Member): Unit = {
    membershipMgmt.deregisterSeed(member.uniqueAddress.address)
  }

  def membershipAdministration: Receive = {
    case MemberUp(member) if member.isMe =>
      clusterStateManagingActor ! SelfAddedToCluster(member)
    case MemberUp(member) =>
      clusterStateManagingActor ! ClusterMemberAdded(member)
    case MemberRemoved(member, _) =>
      clusterStateManagingActor ! ClusterMemberLeft(member)
      deregisterSeedNode(member)
    case LeaderChanged(Some(leader)) if leader != cluster.selfAddress =>
      context.become(membershipListening)
  }

  def membershipListening: Receive = {
    case MemberUp(member) if member.isMe =>
      clusterStateManagingActor ! SelfAddedToCluster(member)
    case MemberUp(member) =>
      clusterStateManagingActor ! ClusterMemberAdded(member)
    case MemberRemoved(member, _) =>
      clusterStateManagingActor ! ClusterMemberLeft(member)
    case LeaderChanged(Some(leader)) if leader == cluster.selfAddress =>
      context.become(membershipAdministration)
  }

  override def receive: Receive = {
    case CurrentClusterState(members, _, _, _, _) =>
      logger.info(s"Received cluster state with $members")
      clusterStateManagingActor ! Initialize
      members.filter(_.status == Up).foreach { member =>
        if (member.isMe) {
          clusterStateManagingActor ! SelfAddedToCluster(member)
        } else {
          clusterStateManagingActor ! ClusterMemberAdded(member)
        }
      }
      context.become(membershipListening)
  }


  private implicit class ExtendedMember(member: Member) {
    def isMe: Boolean = {
      member.uniqueAddress == cluster.selfUniqueAddress
    }

    def isNotMe = !isMe
  }

}
