package com.xebialabs.xlplatform.cluster

import java.util.concurrent.TimeUnit

import com.typesafe.config.Config

import scala.concurrent.duration.{FiniteDuration, _}
import scala.language.postfixOps

sealed trait ClusterMode {
  def configOption: String
}

object ClusterMode {

  // These constants are being use as Spring Profile indicators and therefore need to be Strings and publicly visible.
  final val STANDALONE = "default"
  final val HOT_STANDBY = "hot-standby"
  final val FULL = "full"

  case object Standalone extends ClusterMode {
    val configOption = STANDALONE
  }

  case object HotStandby extends ClusterMode {
    val configOption = HOT_STANDBY
  }

  case object Full extends ClusterMode {
    val configOption = FULL
  }

  private val values = List(Standalone, HotStandby, Full)

  def valueOf(value: String): ClusterMode = values.find(_.configOption == value).getOrElse(
    throw new NoSuchElementException(s"Value [$value] is not a valid xl.cluster.mode option. Possible values: ${values.map(_.configOption).mkString(", ")}."))
}

case class DataSourceConfig(driver: Option[String], url: String, username: String, password: String) {
  override def toString = s"DataSourceConfig($driver, $url, $username)"
}

case class ClusterConfig(config: Config) {

  lazy val mode: ClusterMode = ClusterMode.valueOf(config.getOptionalString("mode").getOrElse(ClusterMode.STANDALONE))

  lazy val name: String = config.getString("name")

  case class Node(hostname: String, clusterPort: Int)

  def node = NodeConfig

  object NodeConfig {

    private lazy val node = {
      val c = config.getConfig("node")
      Node(c.getString("hostname"), c.getInt("clusterPort"))
    }
  }

  object membership {

    private lazy val membership: Config = config.getConfig("membership")

    lazy val heartbeatInterval: FiniteDuration = membership.getDuration("heartbeat", TimeUnit.MILLISECONDS) millis
    lazy val ttl: FiniteDuration = membership.getDuration("ttl", TimeUnit.MILLISECONDS) millis

    lazy val datasource = DataSourceConfig(membership.getOptionalString("jdbc.driver"),
      membership.getString("jdbc.url"),
      membership.getString("jdbc.username"),
      membership.getString("jdbc.password"))
  }

  implicit class ExtendedConfig(config: Config) {
    def getOptionalString(path: String): Option[String] = if (config.hasPath(path)) Some(config.getString(path)) else None

    def getOptionalInt(path: String): Option[Integer] = if (config.hasPath(path)) Some(config.getInt(path)) else None
  }

}
