package com.xebialabs.platform.sso.oidc.user;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.oauth2.core.oidc.IdTokenClaimNames;
import org.springframework.security.oauth2.core.oidc.OidcIdToken;
import org.springframework.security.oauth2.core.oidc.OidcUserInfo;
import org.springframework.security.oauth2.core.oidc.user.OidcUser;
import org.springframework.util.Assert;

/**
 * Custom class which is replica of the {@link org.springframework.security.oauth2.core.oidc.user.DefaultOidcUser}
 * <p>
 * Once https://github.com/spring-projects/spring-security/issues/9366 is addressed, this class can be removed.
 */
public class CustomOidcUser extends CustomOAuth2User implements OidcUser {

    private final OidcIdToken idToken;
    private final OidcUserInfo userInfo;

    /**
     * Constructs a {@code CustomOidcUser} using the provided parameters.
     *
     * @param authorities the authorities granted to the user
     * @param idToken     the {@link OidcIdToken ID Token} containing claims about the user
     */
    public CustomOidcUser(Collection<? extends GrantedAuthority> authorities, OidcIdToken idToken) {
        this(authorities, idToken, IdTokenClaimNames.SUB);
    }

    /**
     * Constructs a {@code CustomOidcUser} using the provided parameters.
     *
     * @param authorities      the authorities granted to the user
     * @param idToken          the {@link OidcIdToken ID Token} containing claims about the user
     * @param nameAttributeKey the key used to access the user's &quot;name&quot; from {@link #getAttributes()}
     */
    public CustomOidcUser(Collection<? extends GrantedAuthority> authorities, OidcIdToken idToken, String nameAttributeKey) {
        this(authorities, idToken, null, nameAttributeKey);
    }

    /**
     * Constructs a {@code CustomOidcUser} using the provided parameters.
     *
     * @param authorities the authorities granted to the user
     * @param idToken     the {@link OidcIdToken ID Token} containing claims about the user
     * @param userInfo    the {@link OidcUserInfo UserInfo} containing claims about the user, may be {@code null}
     */
    public CustomOidcUser(Collection<? extends GrantedAuthority> authorities, OidcIdToken idToken, OidcUserInfo userInfo) {
        this(authorities, idToken, userInfo, IdTokenClaimNames.SUB);
    }

    /**
     * Constructs a {@code CustomOidcUser} using the provided parameters.
     *
     * @param authorities      the authorities granted to the user
     * @param idToken          the {@link OidcIdToken ID Token} containing claims about the user
     * @param userInfo         the {@link OidcUserInfo UserInfo} containing claims about the user, may be {@code null}
     * @param nameAttributeKey the key used to access the user's &quot;name&quot; from {@link #getAttributes()}
     */
    public CustomOidcUser(Collection<? extends GrantedAuthority> authorities, OidcIdToken idToken,
                          OidcUserInfo userInfo, String nameAttributeKey) {
        super(authorities, collectClaims(idToken, userInfo), nameAttributeKey);
        this.idToken = idToken;
        this.userInfo = userInfo;
    }

    private static Map<String, Object> collectClaims(OidcIdToken idToken, OidcUserInfo userInfo) {
        Assert.notNull(idToken, "idToken cannot be null");
        Map<String, Object> claims = new HashMap<>();
        if (userInfo != null) {
            claims.putAll(userInfo.getClaims());
        }
        claims.putAll(idToken.getClaims());
        return claims;
    }

    @Override
    public Map<String, Object> getClaims() {
        return this.getAttributes();
    }

    @Override
    public OidcIdToken getIdToken() {
        return this.idToken;
    }

    @Override
    public OidcUserInfo getUserInfo() {
        return this.userInfo;
    }
}
