package com.xebialabs.platform.sso.oidc.policy.impl;

import com.xebialabs.platform.sso.oidc.exceptions.InvalidRoleClaimsListException;
import com.xebialabs.platform.sso.oidc.policy.ClaimsToGrantedAuthoritiesPolicy;
import org.slf4j.Logger;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.util.Assert;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static java.lang.String.format;
import static org.slf4j.LoggerFactory.getLogger;

/**
 * Provides default behavior for case when a group claim is bound to granted authorities.
 */
public class DefaultClaimsToGrantedAuthoritiesPolicy implements ClaimsToGrantedAuthoritiesPolicy {
    private static final Logger logger = getLogger(DefaultClaimsToGrantedAuthoritiesPolicy.class);
    private final String rolesClaimName;

    public DefaultClaimsToGrantedAuthoritiesPolicy(String rolesClaimName) {
        Assert.hasText(rolesClaimName, "rolesClaimName must contain a property name");
        this.rolesClaimName = rolesClaimName;
    }

    public List<GrantedAuthority> claimsToGrantedAuthorities(Map<String, Object> oidcClaims) {
        List<String> groups;
        if (oidcClaims.get(rolesClaimName) instanceof String) {
            groups = new ArrayList<>();
            groups.add((String) oidcClaims.get(rolesClaimName));
        } else {
            Object list = oidcClaims.getOrDefault(rolesClaimName, new ArrayList<>());
            if (!(list instanceof List)) {
                throw new InvalidRoleClaimsListException(format("Role claims in property [%s] are not a list", rolesClaimName));
            }
            groups = (List) oidcClaims.getOrDefault(rolesClaimName, new ArrayList<>());
        }

        logger.debug("Extracted groups/roles {}", groups);
        try {
            List<GrantedAuthority> authorities = groups.stream().map(g -> new SimpleGrantedAuthority(g)).collect(Collectors.toList());
            return authorities;
        } catch (ClassCastException e) {
            throw new InvalidRoleClaimsListException(format("Role claims name in property [%s] contained non string values", rolesClaimName));
        }
    }
}
