package com.xebialabs.deployit.itest.cloudhost;

import com.google.common.base.Function;
import com.google.common.util.concurrent.ListenableFuture;
import com.xebialabs.overcast.host.CloudHost;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.Socket;

public class WaitForOpenPortTransformer implements CloudHostFutureTransformer {
    private static final Logger logger = LoggerFactory.getLogger(WaitForOpenPortTransformer.class);
    private final int port;
    private final int timeoutSeconds;

    public WaitForOpenPortTransformer(int port, int timeoutSeconds) {
        this.port = port;
        this.timeoutSeconds = timeoutSeconds;
    }

    @Override
    public Function<CloudHost, CloudHost> transform(ListenableFuture<CloudHost> host) {
        return host1 -> {
            logger.info("Waiting for port {}:{}", host1.getHostName(), port);
            waitForPort(host1.getHostName(), port, timeoutSeconds);
            return host1;
        };
    }

    public static void waitForPort(String hostOrIp, int port, int timeoutSeconds) {
        int i = 0;
        while (i < timeoutSeconds) {
            if (isPortOpen(hostOrIp, port)) {
                logger.info("Port {}:{} is open.", hostOrIp, port);
                return;
            }
            try {
                Thread.sleep(1000);
            } catch (InterruptedException e) {
                logger.debug("Interrupted stopping waitForPort {}:{}", hostOrIp, port);
                break;
            }
            i++;
        }
        throw new RuntimeException(String.format("Timeout expired waiting for port %s:%d", hostOrIp, port));
    }

    public static boolean isPortOpen(String hostOrIp, int port) {
        final int timeout = 5000;
        InetSocketAddress endPoint;
        try (Socket socket = new Socket()) {
            endPoint = new InetSocketAddress(InetAddress.getByName(hostOrIp), port);
            if (endPoint.isUnresolved()) {
                return false;
            }
            socket.connect(endPoint, timeout);
            return true;
        } catch (IOException e) {
            logger.debug("Exception testing connection to {}:{}: {}", hostOrIp, port, e.getMessage());
        }
        return false;
    }
}
