package com.xebialabs.xlplatform.tests.runner

import com.xebialabs.deployit.cli.Interpreter
import com.xebialabs.xlplatform.tests.runner.CliScriptsFunSuite.{AroundScriptFinder, TestScriptFilter, TestScriptFinder}
import org.scalatest.funspec.AnyFunSpecLike

import java.io.File
import java.nio.file.Path

object CliScriptsFunSuite {

  import Implicits._

  type AroundScriptFinder = Path => Seq[Path]

  type TestScriptFinder = Option[String] => Seq[Path]

  type TestScriptFilter = Path => Boolean

  val setupScriptFinder: AroundScriptFinder = _.resolveSibling("setup.py").toNilIfNotExists

  val tearDownScriptFinder: AroundScriptFinder = _.resolveSibling("teardown.py").toNilIfNotExists

  val excludeAroundScriptsFilter: TestScriptFilter = path => path.name != "setup.py" && path.name != "teardown.py"
}

trait CliScriptsFunSuite extends CliScriptRunner {
  this: AnyFunSpecLike =>

  import Implicits._

  val testConfig: TestConfig

  private lazy val baseDir: Path = new File(testConfig.runner.baseDirectory)
  private lazy val enabledItests: Seq[String] = (baseDir / "enabled-itests").toNilIfNotExists.flatMap(_.lines)

  lazy val scriptsFinder: TestScriptFinder = {
    case Some(path) if path.trim.nonEmpty => baseDir / path ** "(.+)[.](py|cli)".r
    case _ => baseDir ** "(.+)[.](py|cli)".r
  }

  lazy val scriptFileNameRegexFilter: TestScriptFilter = path => {
    val regex = testConfig.runner.scriptFileNameRegex.r
    path.name match {
      case regex() =>
        true
      case _ =>
        false
    }
  }

  lazy val scriptFileInEnabledListFilter: TestScriptFilter = path => {
    enabledItests.isEmpty || enabledItests.contains(path.name)
  }

  def itShouldRunScripts(scriptFinder: TestScriptFinder = scriptsFinder,
                         scriptFilters: Seq[TestScriptFilter] = Seq(CliScriptsFunSuite.excludeAroundScriptsFilter, scriptFileNameRegexFilter, scriptFileInEnabledListFilter),
                         setupScriptFinder: AroundScriptFinder = CliScriptsFunSuite.setupScriptFinder,
                         teardownScriptFinder: AroundScriptFinder = CliScriptsFunSuite.tearDownScriptFinder, testName: Option[String] = None): Unit = {
    require(baseDir.exists, s"Cannot find path ${baseDir.toAbsolutePath}")
    describe("Script") {
      val allScripts = scriptFinder(testName)
      val filteredScripts = allScripts.filter(script => scriptFilters.forall(_.apply(script)))
      filteredScripts.foreach { script =>
        it(baseDir.relativize(script).path) {
          println("Running " + script.path)
          implicit val interpreter: Interpreter = newInterpreter
          try {
            setupScriptFinder(script).foreach(setup => executeScriptByPath(setup.path))
            executeScriptByPath(script.path)
          } finally {
            teardownScriptFinder(script).foreach(teardown => executeScriptByPath(teardown.path))
          }
        }
      }
    }
  }
}
