package com.xebialabs.xlplatform.test.tasksystem

import java.io.IOException
import java.net.ServerSocket

import akka.actor.ActorSystem
import com.typesafe.config.ConfigFactory
import com.xebialabs.deployit.tasksystem.TaskActorSystem
import org.scalatest.{BeforeAndAfterEach, Suite}

import scala.concurrent.Await.ready
import scala.concurrent.duration._
import scala.util.Random

trait Portfinder {

  def findFreePort(): Int = {
    val socket: ServerSocket = null
    try {
      val socket = new ServerSocket(0)
      socket.setReuseAddress(true)
      val port = socket.getLocalPort
      try {
        socket.close()
      } catch  {
        case e: IOException => // Ignore IOException on close()
      }
      return port
    } catch {
      case e: IOException =>
    } finally {
      if (socket != null) {
        try {
          socket.close()
        } catch {
          case e: IOException =>
        }
      }
    }
    throw new IllegalStateException("Could not find a free TCP/IP port to start Integration Test Server");
  }

}

trait RemoteTestActorSystem extends BeforeAndAfterEach with Portfinder with SecuritySupport {
  this: Suite =>

  //For some of the test that result in calling SatelliteCommunicatorSystem.newActorSystem
  // without overriding default config it's needed to inject random ports via system properties
  System.setProperty("test.port", findFreePort().toString)
  System.setProperty("test.ssl.port", findFreePort().toString)

  initPasswordEncrypter()

  var satelliteClientPort: Int = 0

  implicit var actorSystem: ActorSystem = _

  override protected def beforeEach(): Unit = {
    satelliteClientPort = findFreePort()
    import collection.convert.ImplicitConversionsToJava._
    actorSystem = TaskActorSystem.newActorSystem(ConfigFactory.parseMap(
      Map(
        "xl.satellite.enabled" -> "yes",
        "xl.satellite.timeout.ping" -> "1 second",
        "xl.satellite.timeout.remoteAsk" -> "1 second",
        "xl.server.port" -> satelliteClientPort.toString,
        "xl.algorithms.checksum" -> "SHA-256"
      )), Some(s"${this.getClass.getSimpleName}-task-system-${Random.nextInt(1000)}")
    )

    super.beforeEach()
  }

  override protected def afterEach(): Unit = {
    super.afterEach()
    ready(actorSystem.terminate(), 20.seconds)
  }
}

trait RemoteTestActorSystemConstructor extends Portfinder {
  def defaultConfig: Map[String,Any] = Map(
    "xl.satellite.enabled" -> "yes",
    "xl.satellite.timeout.remoteAsk" -> "1 second",
    "xl.satellite.timeout.ping" -> "1 second",
    "xl.server.port" -> findFreePort()
  )

  def newActorSystemWithConf(customConfig: Map[String, Any]): ActorSystem = {
    val mergedConfig = defaultConfig ++ customConfig
    import collection.convert.ImplicitConversionsToJava._
    TaskActorSystem.newActorSystem(
      ConfigFactory.parseMap(mergedConfig),
      Option(s"${this.getClass.getSimpleName}-task-system-${Random.nextInt(1000)}")
    )
  }
}
