package com.xebialabs.deployit.engine.tasker

import akka.actor.{Actor, ActorLogging, ActorRef, Props}
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState
import com.xebialabs.deployit.engine.tasker.RecoverySupervisorActor.{Delete, Deleted, Write}
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.{StepStateEvent, TaskStateEvent}
import com.xebialabs.deployit.engine.tasker.messages.TaskStateEventHandled

import scala.language.postfixOps

object TaskRecoveryListener {
  def props(recoverySupervisorActor: ActorRef): Props =
    Props(new TaskRecoveryListener(recoverySupervisorActor)).withDispatcher(stateManagementDispatcher)

  val name = "task-recovery-listener"
}

class TaskRecoveryListener(recoverySupervisorActor: ActorRef) extends Actor with ActorLogging {

  import context._

  override def preStart(): Unit = {
    system.eventStream.subscribe(self, classOf[TaskStateEvent])
    system.eventStream.subscribe(self, classOf[StepStateEvent])
    super.preStart()
  }

  def receive: Receive = writerBehaviour

  private def writerBehaviour: Receive = {
    case TaskStateEvent(_, task, oldstate, state)
      if oldstate == TaskExecutionState.UNREGISTERED && state == TaskExecutionState.SCHEDULED =>
      log.debug(s"Task ${task.getId} went from $oldstate to non-final state $state")
      log.debug(s"Skip writing the recovery file for PendingTask ${task.getId}]")
    case TaskStateEvent(_, task, oldstate, state) if !state.isFinal =>
      log.debug(s"Task ${task.getId} went from $oldstate to non-final state $state")
      recoverySupervisorActor ! Write(task)
    case TaskStateEvent(taskId, _, oldState, state) if state.isFinal =>
      log.debug(s"Task $taskId went from $oldState to final state $state")
      recoverySupervisorActor ! Delete(taskId)
      context.become(waitingForDeletionBehaviour(taskId, state, oldState).orElse(writerBehaviour))
    case StepStateEvent(_, _, task, _, _, _, _, _) =>
      log.debug(s"Task ${task.getId} had a step state event")
      recoverySupervisorActor ! Write(task)
  }

  private def waitingForDeletionBehaviour(id: String, state: TaskExecutionState, oldState: TaskExecutionState): Receive = {
    case Deleted(taskId) if taskId == id =>
      log.debug(s"Task $taskId was deleted")
      system.eventStream.publish(TaskStateEventHandled(taskId, oldState, state))
      context.become(writerBehaviour)
  }
}
